# time.pl
# Victor Liu See-le - mailto:victor@n-gon.com
# Helper routines for time.
#
# Copyright (C) 2001 Victor Liu See-le
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version. 
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

use Time::Local;

@DoW = ('Sun','Mon','Tue','Wed','Thu','Fri','Sat');
@MoY = ('Jan','Feb','Mar','Apr','May','Jun',
	    'Jul','Aug','Sep','Oct','Nov','Dec');
$SECS_per_DAY = 24 * 60 * 60;

sub get_iso8601_datetime_utc {
  # Returns the current time in ISO 8601 format, as described in 
  #    http://www.w3.org/TR/NOTE-datetime
  # This is the standard way to encode date/time values acc. to the XBEL DTD.
  
  my ($sec, $min, $hr, $mday, $mon, $yr, $wday, $yday, $isdst) = gmtime(time);
  $mon++; # Zero-based
  $yr += 1900;
  my $s = sprintf("%4d-%02d-%02dT%02d:%02d:%02dZ", 
                  $yr, $mon, $mday, $hr, $min, $sec); 
  return $s;
}

sub time_from_iso8601_utc {

  my $iso8601 = $_[0];
  
  my ($date, $time) = split(/T/, $iso8601);
  
  # $date is of the form YYYY-MM-DD
  my ($YYYY, $MM, $DD) = split(/-/, $date);
  $MM--; # Back to zero-base
  
  # $time is of the form hh:mm:ss.sTZD, where TZD is the Time Zone Designator.
  # The TZD is either 'Z' for UTC, or of the form [+-]hh:mm otherwise.
  # For now, I am assuming the former case. The latter I leave unfinished.
  if ($time =~ /.+Z$/) {
    chop $time;
  } else {
    my ($hhmmsss, $TZD) = split(/[+-]/, $time);
    
    # TODO: Non-UTC case.
    return -1;
  }
  my ($hh, $mm, $ss, $s) = split(/[:\.]/, $time);
  
  return &timegm($ss, $mm, $hh, $DD, $MM, $YYYY);
}

sub formatted_localtime {

  # First arg is format string, second arg is time value.
  # Format string can include the elements:
  #   $YYYY    year
  #   $MM      month
  #   $MO      month of year (ie May, etc.)
  #   $DD      day of month
  #   $WD      day of week (ie Mon, Tue, etc.)
  #   $hh      hour
  #   $mm      minutes
  #   $ss      seconds
  #   $AP      AM/PM designator (omit for military time)
  # An example format string is '$WD $MO $DD, $YYYY at $hh:$mm $AP'
  
  my ($format, $time) = @_;
  my ($sec, $min, $hr, $mday, $mon, $yr, $wday, $yday, $isdst) = localtime($time);
  my ($YYYY, $MM, $MO, $DD, $WD, $hh, $mm, $ss, $AP);
  
  $YYYY = $yr + 1900;
  $MM = $mon + 1;
  $MO = $MoY[$mon];
  $DD = $mday;
  $WD = $DoW[$wday];
  $hh = $hr;
  $mm = sprintf('%02d', $min);
  $ss = sprintf('%02d', $sec);
  
  $_ = $format;
  if (/\$AP/) { 
    # 12-hour format
    if ($hh > 12) {
      $hh -= 12;
      $AP = 'PM';
    } else {
      $AP = 'AM';
    }
    if ($hh == 0) { $hh = 12; }
  }
  s/\$YYYY/$YYYY/g;
  s/\$MM/$MM/g;
  s/\$MO/$MO/g;
  s/\$DD/$DD/g;
  s/\$WD/$WD/g;
  s/\$hh/$hh/g;
  s/\$mm/$mm/g;
  s/\$ss/$ss/g;
  s/\$AP/$AP/g;
  
  return $_;
}

sub elapsed_time_phrase {
  my $since = $_[0];
  my $t = time - $since;
  my $hrs = $t / 3600;
  my $s;
  if ($hrs < 48) {
    $hrs = sprintf('%d', $hrs); $hrs++;
    $s = ($hrs == 1) ? "less than 1 hour ago" : "less than $hrs hours ago";
  } else {
    $days = sprintf('%d', ($hrs / 24));
    $s = ($days == 1) ? "more than 1 day ago" : "$days days ago";
  }
  return $s;
}
1;
