#! /usr/bin/perl

# b.mod2.pl 
# Victor Liu See-le - mailto:victor@n-gon.com
# Via HTML form, add or modify bookmarks in XBEL format.
#
# Information on the XML Bookmark Exchange Language (XBEL) is at:
#        http://www.python.org/topics/xml/xbel/
#
# Copyright (C) 2001 Victor Liu See-le
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version. 
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

require "b.parse.pl";
require "cgi.pl";
require "time.pl";


###############################################################################
# Begin MAIN                                                                  #

&validate_user;
&register_handlers;
&get_form_data;
&open_and_lock_xml_file;
&process_xbel;
&abort_if_exceed_shr_limit if $care_about_shr_stats;
&move_item unless $this_is_where_we_want_to_be;
&write_xml_with_backup;
&unlock_and_close_xml_file;
&redisplay($src, $view_folder);

exit(0);

# End of MAIN                                                                 #
###############################################################################

sub register_handlers {
  &set_handler("Element-Start.folder", \&handle_folder_start);
  &set_handler("Element-Start.bookmark", \&handle_bookmark_start);
  &set_handler("Element-Start.comment", \&handle_comment_start);
  &set_handler("Element-End.xbel", \&handle_xbel_end);
  &set_handler("Element-End.folder", \&handle_folder_end);
  &set_handler("Element-End.bookmark", \&handle_bookmark_end);
  &set_handler("Element-End.title", \&handle_title_end);
  &set_handler("Element-End.desc", \&handle_desc_end);
  &set_handler("Text.folder.title", \&handle_folder_title);
  &set_handler("Text.folder.desc", \&handle_folder_desc);
  &set_handler("Text.bookmark.title", \&handle_bookmark_title);
  &set_handler("Text.bookmark.desc", \&handle_bookmark_desc);
}

sub handle_folder_start {
  if (! $found_modify_item) {
    &look_for_modify_item;
  } else {
    &finish_modify_item(&get_index-1);
  }
  
  return if $append; # Done; don't need to check for destination folder.
  
  # Check if this is the destination folder.
  my $id = &get_id;
  if ($id eq $dest_folder) {

    # Record the level. Will need this to mark the end of this folder.
    $dest_folder_level = &get_depth;
  }
}

sub handle_bookmark_start {
  if (! $found_modify_item) {
    &look_for_modify_item;
  } else {
    &finish_modify_item(&get_index-1);
  }
  &update_shr_stats if $care_about_shr_stats;
}

sub handle_comment_start {
  # If we've found the item to modify, record highest id number for each
  # comment we encounter in this item. We'll need this to create a new id.
  return if ($new_comment and ! $found_modify_item);
  my $id = &get_id;
  if ($id =~ /-comment\.(\d+)$/) {
    if ($highest_comment_idnum < $1) {
      $highest_comment_idnum = $1;
    }
  }
}

sub handle_xbel_end {
  return if $append; # Don't need to check for destination folder.
  
  # For the case that the destination folder is top-level, mark
  # this position as where to plop the modified item.
  if ($dest_folder eq 'folder.0') {
    $dest_folder_end = &get_index;
  }
}

sub handle_folder_end {
  &look_for_modify_item_end if $found_modify_item;

  return if $append; # Don't need to check for destination folder.
  
  # If we already have found the beginning of the destination 
  # folder, check if this is the end tag corresponding to it. 
  # If so, this position is the end of the destination folder.
  my $l = &get_depth;
  if ((! $dest_folder_end) and 
      ($dest_folder_level) and 
      ($dest_folder_level == $l)) {
    $dest_folder_end = &get_index;
  }
}

sub handle_bookmark_end {
  &look_for_modify_item_end if $found_modify_item;
}

sub handle_title_end {
  return if $append;
  
  # If flag is set, then modify info. 
  # This handles the case for <title></title>.
  if ($found_modify_item and $need_to_write_title) {
    my $i = &get_index;
    &splice_xbel($i, 0, $new_title);
    $need_to_write_title = 0;
  }
}

sub handle_desc_end {
  return if $append;
  
  # If flag is set, then modify info. 
  # This handles the case for <desc></desc>.
  if ($found_modify_item and $need_to_write_desc) {
    my $i = &get_index;
    &splice_xbel($i, 0, $new_desc);
    $need_to_write_desc = 0;
  }
}

sub handle_folder_title {
  return if $append;
  
  # If flag is set, then modify info.
  # Note: Check that $need_to_write_title is set, since this ensures
  # that $new_title is not an empty string. Ie, must have a title!
  if ($found_modify_item and $need_to_write_title) {
    my $i = &get_index;
    &splice_xbel($i, 1, $new_title);
    $need_to_write_title = 0;
  }
}

sub handle_folder_desc {
  return if $append;
  
  # If flag is set, then modify info.
  if ($found_modify_item) {
    # Splice new description into XML.
    my $i = &get_index;
    &splice_xbel($i, 1, $new_desc);
    # Clear flag
    if ($need_to_write_desc) {
      $need_to_write_desc = 0;
    }
  }
}

sub handle_bookmark_title {
  &handle_folder_title;
}

sub handle_bookmark_desc {
  &handle_folder_desc;
}

sub get_form_data {
  &parse_form_data(*dat);
  $modify_item_id = $dat{'id'}; # If this doesn't exist, then adding new item.
  $view_folder = $dat{'view_folder'};
  $src = $dat{'src'};
  $new_type = $dat{'add_type'};
  $new_uri = $dat{'uri'};
  $new_title = &encode_html_entities($dat{'title'}) and 
      $need_to_write_title = 1;
  $new_desc = &encode_html_entities(&trim($dat{'desc'})) and 
      $need_to_write_desc = 1;
  $dest_folder = $dat{'dest_folder'} ? $dat{'dest_folder'} : 'folder.0';
  $dest = $dat{'dest'};
  if ($dest && $dest eq 'shr') {
    $destination_xml_file = $shr_dir . $shr_xml_file;
    &set_working_xml_file($destination_xml_file);
    $care_about_shr_stats = (($shr_on) and ($shr_limit) and (defined $user));
  }
  else {
    $destination_xml_file = $xml_file;
  }
  $append = $dat{'append'};
  $new_comment = &encode_html_entities(&trim($dat{'comment'}));
}

sub look_for_modify_item {
  my $type = &get_type;
  my $i = &get_index;
  my %att = &parse_attributes(&get_attributes);
  my $id = $att{'id'};

  # Check if this is the item to modify.
  if (($modify_item_id) and ($modify_item_id eq $id)) {

    # This is it. Modify items, and set flag to keep looking for other 
    # attributes.
    $modify_item_type = $type;
    $modify_item_attributes = &get_attributes;
    if ($type eq "bookmark" and ! $append) {
      my $dt = &get_iso8601_datetime_utc;
      $modify_item_attributes = &set_attribute($modify_item_attributes, 
                                               "href", $new_uri);
      $modify_item_attributes = &set_attribute($modify_item_attributes, 
                                               "modified", $dt);
      my $s = qq[<$type $modify_item_attributes>];
      &splice_xbel($i, 1, $s);
    }
    $modify_item_level = &get_depth;
    $mark_modify_item_begin = $i;
    $found_modify_item = 1;

    # Check to see if we're already in the destination folder.
    if ($append or
        (&get_parents_type eq "folder" and 
            $dest_folder eq &get_parents_id) or
        (&get_parents_type eq "xbel" and 
            $dest_folder eq 'folder.0'))
    {
      $this_is_where_we_want_to_be = 1;
    }

  }
}

sub look_for_modify_item_end {
  my $type = &get_type;
  my $i = &get_index;
  my $l = &get_depth;

  # If beginning mark is set for the modified tag, look for end.
  # Write any items which still need to be written.
  if ((! $mark_modify_item_end) and 
      ($mark_modify_item_begin) and 
      ($modify_item_type eq $type) and 
      ($l == $modify_item_level)) 
  {
    $mark_modify_item_end = $i;
    &finish_modify_item($i-1);
  }
}

sub finish_modify_item {
  my $location_to_splice = $_[0];
  return if (! defined $location_to_splice or $location_to_splice < 0);

  # Clear flag if set.
  if ($found_modify_item) {
    $found_modify_item = 0;

    # Further, write remaining items.
    if ($append) {
      if ($append eq 'comment' and $new_comment) {
        my $comment_idnum = $highest_comment_idnum + 1;
        my $dt = &get_iso8601_datetime_utc;
        my $s = qq[\n<comment id="$modify_item_id-comment.$comment_idnum"];
        $s .= qq[ added="$dt"];
        $s .= qq[ contributer="$user"] if $user;
        $s .= qq[>$new_comment</comment>];
        &splice_xbel($location_to_splice, 0, $s);
      } 
    }
    else {
      if ($need_to_write_title) {
        my $s = qq[\n<title>$new_title</title>];
        &splice_xbel($location_to_splice, 0, $s);
        $need_to_write_title = 0;
      }
      if ($need_to_write_desc) {
        my $s = qq[\n<desc>$new_desc</desc>];
        &splice_xbel($location_to_splice, 0, $s);
        $need_to_write_desc = 0;
      }
    }
  }
}

sub move_item {

  # Splice out modified item. Splice back in at end of destination folder.
  my @item;

  if ($modify_item_id) {
    my $len = $mark_modify_item_end - $mark_modify_item_begin + 1;
    if ($len > 0) {
      @item = &splice_xbel($mark_modify_item_begin, $len);
      $dest_folder_end -= $len if ($mark_modify_item_begin < $dest_folder_end);
    }
  }
  else {
    # No modified item exists, so create new item.
    my $s;
    my $idnum = &get_next_idnum;
    my $dt = &get_iso8601_datetime_utc;
    
    $s = qq[<$new_type];
    if ($new_type eq 'bookmark') {
      $s .= qq[ href="$new_uri"];
    }
    $s .= qq[ id="$new_type.$idnum" added="$dt"];
    if ($new_type eq 'bookmark') {
      $s .= qq[ modified="$dt"];
      $s .= qq[ hits="0"] if $dest eq 'shr';
    }
    if (defined $user and $dest eq 'shr') {
      $s .= qq[ contributer="$user"];
    }
    $s .= ">\n";
    push(@item, $s); 
    push(@item, "<title>$new_title</title>\n");
    if ($new_desc) {
      push(@item, "<desc>$new_desc</desc>\n");
    }
    push(@item, "</$new_type>\n");
  }
  &splice_xbel($dest_folder_end, 0, @item);
}


