# b.display.pl
# Victor Liu See-le - mailto:victor@n-gon.com
# Helper routines to parse an XBEL format document and produce HTML display.
#
# Information on the XML Bookmark Exchange Language (XBEL) is at:
#        http://www.python.org/topics/xml/xbel/
#
# Copyright (C) 2001 Victor Liu See-le
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version. 
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

require "b.parse.pl";
require "time.pl";


sub store_open_folders {
  &set_handler("Element-Start.folder", \&look_for_open_folder);
  &open_and_lock_xml_file;
  &process_xbel;
  &write_xml_with_backup if &xml_has_changed;
  &unlock_and_close_xml_file;
  &reset_xbel;
}

sub look_for_open_folder {
  my $id = &ensure_valid_id(&get_id);

  # There are different display modes depending on the value of view_folder
  # parameter. 
  #    1. If view_folder=all, then all folders are open.
  #    2. If view_folder=<folder_id>, then folder with this folder_id, and
  #       all enclosing folders, are open. 
  #    3. If view_folder is not present, then all folders without attribute
  #       folded=no are open. This is the default behavior.
  
  if ($view_folder eq 'all') {
    push(@open_folders, $id);
  }
  elsif ($view_folder && ($id eq $view_folder or $view_folder eq 'folder.0')) { 
    push(@open_folders, $view_folder);
  }
  elsif (! $view_folder) {
    my %att = &parse_attributes(&get_attributes);
    my $folder_att = $att{'folded'} ? $att{'folded'} : '';
    my $is_folded = (($folder_att ne '') && ($folder_att ne 'no'));
    
    unless ($is_folded) {
    
      # This folder may be open, but all parent folders must also be open in
      # order for this folder to be included in @open_folders.
      my $l = &get_depth;
      foreach my $i (1 .. $l) {
        if (&get_node_type($i) eq 'folder') {
          my $node_id = &get_node_id($i);
          return if ! grep(/^$node_id$/, @open_folders);
        }
      }
          
      push(@open_folders, $id);
    }  
  }  
}

sub display_xbel_start { 
  &print_div_open('xbel-cl');
  print "\n";
}

sub display_folder_start {

  # If we encounter a subitem, must remember to close enclosing folder.
  if ($flag_display_item) {
    $need_to_close_folder_level[(&get_folder_level)-1] = 1;
  }
  
  &display_if_in_open_folder;
}

sub display_bookmark_start {

  # If we encounter a subitem, must remember to close enclosing folder.
  if ($flag_display_item) {
    $need_to_close_folder_level[&get_folder_level] = 1;
  }
  
  &display_if_in_open_folder;
}

sub display_comment_start {
  return if ! $flag_display_item;
  my $type = &get_parents_type . '-comment';
  my $id = &get_id;
  my %att = &parse_attributes(&get_attributes);
  my $contributer = $att{'contributer'};
  
  &print_div_open("$type-cl");
  
  # Display user buttons for this item. 
  if ($show_user_buttons) {
    
    # Delete comment button
    if ($is_user_admin or ($user_can_delete_comment and
                           defined $user and
                           $user eq $contributer))
    { 
        
      my $bshrdel_path = $bdel_path . "\?id=$id\&src=shr";
      if ($view_folder and $view_folder ne 'all') {
        $bshrdel_path .= "\&view_folder=$view_folder";
      }
          
      &print_div_open("$type-del-btn-cl");
      print qq[<span class="$type-del-btn-text-cl">];
      print qq[<a href="$bshrdel_path" title="Delete this item">];
      print qq[shr:del</a></span>];
      &print_div_close;
    }
  }
      
  $after_comment_open_div = &xtra_info_div('open', $type, %att);
  $after_comment_text_div = &xtra_info_div('text', $type, %att);
  
  if ($after_comment_open_div) {
    print $after_comment_open_div;
    $after_comment_open_div = '';
  }
}

sub display_info_start {
  return if (&get_parents_type ne 'xbel');
  &print_div_open('info-cl');
}

sub display_xbel_end {
  &print_div_close;
}

sub display_folder_end {
  my $need_to_close = 1 if $flag_display_item;
  &finish_previous_item;
  if ($need_to_close) {
    &print_div_close;
  } else {
    my $fl = &get_folder_level;
    if ($need_to_close_folder_level[$fl]) {
      $need_to_close_folder_level[$fl] = 0; # Reset.
      &print_div_close;
    }
  }
}

sub display_bookmark_end {
  my $need_to_close = 1 if $flag_display_item;
  &finish_previous_item;
  &print_div_close if $need_to_close;
}

sub display_comment_end {
  &print_div_close if $flag_display_item;
}

sub display_info_end {
  return if (&get_parents_type ne 'xbel');
  &print_div_close;
}

sub display_xbel_title {
  my $title = &get_text;
  print qq[<h1>$title</h1>\n];
}

sub display_folder_title {
  return if ! $flag_display_item;

  my $id = &get_parents_id;
  my $title = &get_text;
  if (! $title) { $title = "Untitled"; }

  &print_div_open('folder-title-cl');

  if ($interactive_folding) { 
    if (&is_open_folder($id)) {
      print qq[<a href="$bfold_path\?id=$id\&fold=yes"];
      print qq[ title="Close this folder">];
      if ($folder_open_icon) {
        print qq[<img src="$theme_path/$folder_open_icon" alt="-" /></a>];
      } else {
        print qq[-</a> ];
      }
    } else {
      print qq[<a href="$bfold_path\?id=$id\&fold=no"];
      print qq[ title="Open this folder">];
      if ($folder_icon) {
        print qq[<img src="$theme_path/$folder_icon" alt="+" /></a>];
      } else {
        print qq[+</a> ];
      }
    }
    print qq[$title];
  }
  elsif ($view_folder eq 'all') {
    if ($folder_open_icon) {
      print qq[<img src="$theme_path/$folder_open_icon" alt="" />];
    }
    print qq[$title];
  }
  elsif ($view_folder) {
    if ($view_folder ne $id) {
      my $href = "$bsee_path\?view_folder=$id";
      $href .= "\&src=shr" if $src eq 'shr';

      if (&is_open_folder($id)) {
        if ($folder_open_icon) {
          print qq[<a href="$href" title="View this folder">];
          print qq[<img src="$theme_path/$folder_open_icon" alt="" />];
          print qq[</a>];
        }
      } else {
        if ($folder_icon) {
          print qq[<a href="$href" title="View this folder">];
          print qq[<img src="$theme_path/$folder_icon" alt="" />];
          print qq[</a>];
        }
      }
      print qq[<a href="$href" title="View this folder">$title</a>];
    } 
    else {
      if ($folder_open_icon) {
        print qq[<img src="$theme_path/$folder_open_icon" alt="" />];
      }
      print qq[$title];
    }
  }
  &print_div_close;
  
  if ($after_title_div) {
    print $after_title_div;
    $after_title_div = '';
  }
}

sub display_bookmark_title {
  return if ! $flag_display_item;
  
  my $id = &get_parents_id;
  my $title = &get_text;
  my %att = &parse_attributes(&get_parents_attributes);
  my $href = $att{'href'};
  my $help_text = "Link to $href";
  if (! $title) { $title = "Untitled"; }

  # Do we record hits and last access times for this bookmark?
  if ($record_hits) {
    $href = "$blink_path\?id=$id";
    if ($src eq 'shr') {
      $href .= "\&src=shr";
    }
  }
  
  &print_div_open('bookmark-title-cl');
  print qq[<a href="$href" title="$help_text"];
  print qq[ target="_new"] if $links_appear_in_new_window;
  print qq[>];
  if ($bookmark_icon) {
    print qq[<img src="$theme_path/$bookmark_icon" alt="&gt;" />];
  }
  print qq[$title</a>];
  &print_div_close;
  
  if ($after_title_div) {
    print $after_title_div;
    $after_title_div = '';
  }
}

sub display_xbel_desc {
  my $desc = &get_text;
  print qq[$desc\n];
}

sub display_folder_desc {
  if ($flag_display_item and (my $text = &get_text)) {
    &print_div_open('folder-desc-cl');
    print $text;
    &print_div_close;
  }
  if ($after_desc_div) {
    print $after_desc_div;
    $after_desc_div = '';
  }
}

sub display_folder_comment {
  if ($flag_display_item and (my $text = &get_text)) {
    &print_div_open('folder-comment-text-cl');
    print $text;
    &print_div_close;
    if ($after_comment_text_div) {
      print $after_comment_text_div;
      $after_comment_text_div = '';
    }
  }
}

sub display_bookmark_desc {
  if ($flag_display_item and (my $text = &get_text)) {
    &print_div_open('bookmark-desc-cl');
    print $text;
    &print_div_close;
  }
  if ($after_desc_div) {
    print $after_desc_div;
    $after_desc_div = '';
  }
}

sub display_bookmark_comment {
  if ($flag_display_item and (my $text = &get_text)) {
    &print_div_open('bookmark-comment-text-cl');
    print $text;
    &print_div_close;
    if ($after_comment_text_div) {
      print $after_comment_text_div;
      $after_comment_text_div = '';
    }
  }
}

sub display_metadata {
  my $t = &get_text_indent;
  my $type = &get_node_type(2);

  return if (&get_parents_type ne 'info' or ! $type);

  if ($type eq 'xbel') {
    my %att = &parse_attributes(&get_attributes);
    my $key;
    $t .= $tab;
    foreach $key (sort keys(%att)) { 
      print qq[$t$key: $att{$key}<br />\n]; 
    }
  }
}

sub display_if_in_open_folder {
  &finish_previous_item;

  my $id = &ensure_valid_id(&get_id);
  
  # If we're viewing by_folder, and we're below top-level, then there's a 
  # one-time thing we've gotta do before printing any folders and bookmarks,
  # and that is to display the links to the top-level folder and to the 
  # parent folder.
  if ($view_folder and $view_folder eq $id) {
    &display_top_and_up_links;
  }
  
  # If we're in an open folder, start saving info for this item.
  if (&is_in_open_folder or $id eq $view_folder) {
    my $type = &get_type;
    my %att = &parse_attributes(&get_attributes);
    
    &print_div_open("$type-cl");
    print "\n";
    
    # Display user buttons for this item. 
    if ($show_user_buttons) {
    
      if ($src eq 'shr') { # Shared bookmarks have different buttons.
        
        my $xtra_params;
        $xtra_params = "\&src=$src" if $src;
        $xtra_params .= "\&view_folder=$view_folder" if $view_folder;
        
        &print_div_open("$type-btn-cl");
        print qq[<span class="$type-btn-text-cl">];
        print qq[<a href="$bmod_path\?id=$id$xtra_params" ];
        print qq[title="Add this item to my bookmarks">add</a></span>];
        &print_div_close;
      
        &print_div_open("$type-contrib-btn-cl");
        print qq[<span class="$type-contrib-btn-text-cl">];
        print qq[<a href="$bcontrib_path\?id=$id$xtra_params" ];
        print qq[title="Contribute a comment">contrib</a></span>];
        &print_div_close;
        
        # Contributer of shared item can delete.
        my %att = &parse_attributes(&get_attributes);
        my $contributer = $att{'contributer'};
        if (defined $user and ($contributer eq $user or $is_user_admin)) {
        
          my $bshrmod_path = $bmod_path . "\?id=$id\&dest=shr$xtra_params";
          my $bshrdel_path = $bdel_path . "\?id=$id$xtra_params";
          
          &print_div_open("$type-shr-btn-cl");
          print qq[<span class="$type-shr-btn-text-cl">];
          print qq[<a href="$bshrmod_path" title="Modify this item">];
          print qq[shr:mod</a>];
          print qq[ | ];
          print qq[<a href="$bshrdel_path" title="Delete this item">];
          print qq[shr:del</a></span>];
          &print_div_close;
        }
      }
      else { # Private bookmarks
        
        # Add link to b.mod and b.del
        &print_div_open("$type-btn-cl");
        print qq[<span class="$type-btn-text-cl">];
        print qq[<a href="$bmod_path\?id=$id" title="Modify this item">];
        print qq[mod</a></span>];
        print qq[ | ];
        print qq[<span class="$type-btn-text-cl">];
        print qq[<a href="$bdel_path\?id=$id" title="Delete this item">];
        print qq[del</a></span>];
        &print_div_close;
      
        # If sharing is turned on, add link for exporting to shares.
        if ($shr_on and $src ne 'shr') {
          my $bshradd_path = "$bmod_path?id=$id\&dest=shr";
          print qq[<div class="$type-shr-btn-cl">\n];
          print qq[<span class="$type-shr-btn-text-cl">];
          print qq[<a href="$bshradd_path" title=];
          print qq["Add this item to shared bookmarks">shr:add</a></span>\n];
          print qq[</div>\n];
        }
      }
    }
    
    # Extra information on this item can be displayed after the item's title
    # or desc. Which extra info, and the order in which they're displayed,
    # are specified in b.conf (cf after_bookmark_title_display_order, etc).
    $after_open_div = &xtra_info_div('open', $type, %att);
    $after_title_div = &xtra_info_div('title', $type, %att);
    $after_desc_div = &xtra_info_div('desc', $type, %att);
    
    # Go ahead and print the $after_open_div stuff.
    if ($after_open_div) {
      print $after_open_div;
      $after_open_div = '';
    }
    
    $flag_display_item = 1;
    
  }
}

sub xtra_info_div {
  my ($after_where, $type, %att) = @_; # $after_where = title | desc | open
  my $lookup_type = ($src eq 'shr') ? 'shr_' . $type : $type; 
  $lookup_type =~ s/-/_/;
  my $order_str = eval '$after_' . $lookup_type . '_' . $after_where . 
                       '_display_order';
  return if ! $order_str or ! $after_where;
  my @order = split(/[, ]/, $order_str);
  
  my $div = qq[\n<div class="$type-after-$after_where-cl">\n];
  foreach my $x (@order) {
    # Each $x is part of the name of a subroutine below. For example,
    # if $x = 'added', the subroutine &added_div is called.
    $div .= eval '&' . $x . "_div('$type', %att)";
    if ($@) { # Error, abort.
      print $@;
      &print_html_footer;
      exit -1;
    }
  }
  $div .= qq[</div>\n];

  return $div;
}

sub added_div {
  my ($type, %att) = @_;
  my $added = $att{'added'};
  my $contributer = $att{'contributer'};
  return if ! ($added or $contributer);
  
  my $div = qq[<div class="$type-added-cl">Added];
  if ($contributer) {
    my $busrnfo_params = "?usrnfo_on=$contributer";
    $busrnfo_params .= "\&src=$src" if $src;
    $busrnfo_params .= "\&view_folder=$view_folder" if $view_folder;
    $div .= qq[ by <a href="$busrnfo_path$busrnfo_params"];
    $div .= qq[ title="Get info on $contributer"];
    $div .= qq[>$contributer</a>];
  }
  if ($added) {
    my $t = &time_from_iso8601_utc($added);
    my $s = &formatted_localtime($datetime_format, $t);
    $div .= qq[ on $s];
  }
  $div .= qq[</div>\n];

  return $div;
}

sub modified_div {
  my ($type, %att) = @_;
  my $modified = $att{'modified'};
  my $contributer = $att{'contributer'};
  return if ! ($modified or $contributer);
  
  my $div = qq[<div class="$type-modified-cl">Modified];
  if ($contributer) {
    my $busrnfo_params = "?usrnfo_on=$contributer";
    $busrnfo_params .= "\&src=$src" if $src;
    $busrnfo_params .= "\&view_folder=$view_folder" if $view_folder;
    $div .= qq[ by <a href="$busrnfo_path$busrnfo_params"];
    $div .= qq[ title="Get info on $contributer"];
    $div .= qq[>$contributer</a>];
  }
  if ($modified) {
    my $t = &time_from_iso8601_utc($modified);
    my $s = &formatted_localtime($datetime_format, $t);
    $div .= qq[ on $s];
  }
  $div .= qq[</div>\n];

  return $div;
}

sub hitsperday_indicator_div {
  my ($type, %att) = @_;
  my $added = $att{'added'};
  my $hits = $att{'hits'};
  return if ! $added or ! defined $hits;
  
  my $t = time - &time_from_iso8601_utc($added);
  my $days = $t / $SECS_per_DAY; $days = 0.1 if ! $days;
  my $hpd = $hits / $days;
  my $meter = 0;
  for (my $i=1; ; $i++) {
    my $threshold = eval("\$hitsperday_indicator_threshold_$i");
    if ($threshold and $hpd >= $threshold) {
      $meter++;
    } else {
      last;
    }
  }
  my $img = eval("\$hitsperday_indicator_icon_$meter");
  my $hpd_str = sprintf($hitsperday_label, $hpd); 
  my $div = qq[<div class="bookmark-hitsperday-cl">];
  if ($img) {
    $div .= qq[<img src="$theme_path/$img" title="$hpd_str" />];
  } else {
    $div .= $hpd_str;
  }
  $div .= qq[</div>\n];
  
  return $div;
}
  
sub visited_div {
  my ($type, %att) = @_;
  my $visited = $att{'visited'};
  return if ! $visited;
  
  my $t = &time_from_iso8601_utc($visited);
  my $s = &formatted_localtime($datetime_format, $t);
  my $div = qq[<div class="$type-visited-cl">Last visited on $s</div>\n];
  
  return $div;
}
  
sub visited_elapsed_time_div {
  my ($type, %att) = @_;
  my $visited = $att{'visited'};
  return if ! $visited;
  
  my $s = &elapsed_time_phrase(&time_from_iso8601_utc($visited));
  my $div = qq[<div class="$type-visited-cl">Last visited $s];
  $div .= qq[</div>\n];

  return $div;
}

sub comment_contributer_div {
  my ($type, %att) = @_;
  my $contributer = $att{'contributer'};
  return if ! $contributer;
  
  my $busrnfo_params = "?usrnfo_on=$contributer";
  my $div = qq[<div class="$type-contributer-cl">];
  $div .= qq[- <a href="$busrnfo_path$busrnfo_params"];
  $div .= qq[ $type title="Get info on $contributer"];
  $div .= qq[>$contributer</a>];
  $div .= qq[</div>\n];

  return $div;
}

sub comment_added_div {
  my ($type, %att) = @_;
  my $added = $att{'added'};
  return if ! $added;
  
  my $t = &time_from_iso8601_utc($added);
  my $s = &formatted_localtime($datetime_format, $t);
  my $div = qq[<div class="$type-added-cl">];
  $div .= qq([$s]);
  $div .= qq[</div>\n];

  return $div;
}

sub display_top_and_up_links {
  &print_div_open('folder-cl');
  print "\n";
  &print_div_open('folder-title-cl');
  my $href = "$bsee_path\?view_folder=folder.0";
  $href .= "\&src=shr" if $src eq 'shr';
  print qq[<a href="$href" title="Go to top-level">];
  print qq[<img src="$theme_path/$folder_icon" alt="" />] if ($folder_icon);
  print qq[(top-level)</a>];
  &print_div_close;
  &print_div_close;
  
  my $parents_id = &get_parents_id;
  return if (! $parents_id);
  &print_div_open('folder-cl');
  print "\n";
  &print_div_open('folder-title-cl');
  $href = "$bsee_path\?view_folder=$parents_id";
  $href .= "\&src=shr" if $src eq 'shr';
  print qq[<a href="$href" title="Go up one level">];
  print qq[<img src="$theme_path/$folder_icon" alt="" />] if ($folder_icon);
  print qq[(up)</a>];
  &print_div_close;
  &print_div_close;
}

sub is_open_folder {
  my $id = $_[0];
  $id =~ s/(\W)/\\$1/g; # Escape non-word chars.
  my $is_open = grep(/^$id$/, @open_folders);
  return $is_open;
}

sub is_in_open_folder {

  # view_folder=all
  return 1 if ($view_folder eq 'all');

  # Top level, always visible in interactive mode, only visible if we're
  # looking at the top-level folder in by_folder mode.
  return 1 if (&get_depth == 1 and (! $view_folder or 
                                    $view_folder eq 'folder.0'));

  # Otherwise, check if parent is a member of @open_folders.
  my $parents_id = &get_parents_id;
  my $is_open = $parents_id && grep(/^$parents_id$/, @open_folders);
  return $is_open;
}

sub finish_previous_item {
  # This is called either when a folder or bookmark is closed, or when a 
  # new item is reached in the xbel document. Do stuff to finish off the 
  # previous folder or bookmark, including clearing flags, etc.
  $flag_display_item = 0;
  
  # If the previous item did not have a title or desc, then these extra
  # info divs were not printed.
  if ($after_title_div) {
    print $after_title_div;
    $after_title_div = '';
  }
  if ($after_desc_div) {
    print $after_desc_div;
    $after_desc_div = '';
  }
}
  
sub inactive_menu_item {
  my $inactive = ($src eq 'shr') ? 'shr:see' : 'see';
  return $inactive;
}

sub print_div_open {
  my $div_class = $_[0];
  print qq[<div class="$div_class">];
}

sub print_div_close {
  print qq[</div>\n];
}

1;