VERSION 1.0 CLASS
BEGIN
  MultiUse = -1  'True
END
Attribute VB_Name = "clsFileIO"
Attribute VB_Creatable = False
Attribute VB_Exposed = False
'**************************************************************************
' FILEIO.CLS - A wrapper class for simple binary file input/output.
'**************************************************************************
Option Explicit
'--------------------------------------------------------------------------
' Class-specific private variables.
'--------------------------------------------------------------------------
Private mintFileHandle As Integer   ' Keeps track of the file handle.
Private mstrFileName As String      ' Holds the file name and path.
Private mblnFileOpen As Boolean     ' Flag used to determine if the current
                                    ' file is open.
Private mblnReadOnly As Boolean     ' Holds the current mode for the file.

'**************************************************************************
' Purpose:  Read/Write Property that determines the maximum number of
'           characters to read into a textbox.
' Effects:  LoadTextBox, Read, ReadFileIntoArray, WriteArrayToFile methods.
' Assumes:  That the value in this variable will not cause an overflow if
'           text is loaded into a textbox or string array element.
'**************************************************************************
Public TextBoxLimit As Double
'**************************************************************************
' Purpose:  To retrieve the file handle being used by the current instance
'           of the class.
' Outputs:  A file handle from 1 to 512.
' Assumes:  Class_Initialize was able to get a free file handle.
'**************************************************************************
Public Property Get FileHandle() As Integer
    FileHandle = mintFileHandle
End Property
'**************************************************************************
' Purpose:  To retrieve the name of the currently open file.
' Outputs:  A file name with extension (no path).
' Assumes:  OpenFile has been called.
'**************************************************************************
Public Property Get FileName() As String
    FileName = GetFileInfo(mstrFileName, False)
End Property
'**************************************************************************
' Purpose:  To determine if OpenFile was called sucessfully.
' Outputs:  The current file open status.
'**************************************************************************
Public Property Get FileOpen() As Boolean
    FileOpen = mblnFileOpen
End Property
'**************************************************************************
' Purpose:  To return the length of the currently open file.
' Outputs:  The size of the current file, in bytes.
'**************************************************************************
Public Property Get FileLength() As Double
    If mblnFileOpen Then
        FileLength = LOF(mintFileHandle)
    Else
        Err_FileNotOpen
    End If
End Property
'**************************************************************************
' Purpose:  To determine the current read/write position within the file.
' Outputs:  The current file pointer position.
'**************************************************************************
Public Property Get CurrentPosition() As Double
    If mblnFileOpen Then
        CurrentPosition = Seek(mintFileHandle)
    Else
        Err_FileNotOpen
    End If
End Property
'**************************************************************************
' Purpose:  To initialize class-level variables.
' Effects:  Loads mintFileHandle with a free file handle, and sets the
'           initial limit for the TextBoxLimit property.
'**************************************************************************
Private Sub Class_Initialize()
    Const TOO_MANY_FILES_OPEN = 67  ' Provides a readable error value.
    Dim strErrMsg As String         ' Holds error message, if necessary.

    On Error Resume Next            ' Resume on all errors.
    '---------------------------------------------------------------------
    ' Get a free file handle.
    '---------------------------------------------------------------------
    mintFileHandle = FreeFile
    '---------------------------------------------------------------------
    ' If FreeFile fails, then more than 255 files are open. Try to call
    ' FreeFile again to obtain a file handle between 256 and 512
    ' (inclusive).
    '---------------------------------------------------------------------
    If Err = TOO_MANY_FILES_OPEN Then
        Err.Clear   ' Reset the error object.
        mintFileHandle = FreeFile(1)
        '-----------------------------------------------------------------
        ' Notify the user that too many files are open.
        '-----------------------------------------------------------------
        If Err Then
            strErrMsg = "No more file handles are available. Please close"
            strErrMsg = strErrMsg & " an open file and try again!"
            MsgBox strErrMsg, vbCritical
        End If
    End If
    '---------------------------------------------------------------------
    ' Use a safe default limit for TextBoxLimit.
    '---------------------------------------------------------------------
    TextBoxLimit = 44000
End Sub
'**************************************************************************
' Purpose:  Close the open file.
' Effects:  Closes the file pointed to by mintFileHandle.
'**************************************************************************
Private Sub Class_Terminate()
    Close mintFileHandle
End Sub
'**************************************************************************
' Purpose:  Determines if a file exists.
' Inputs:   The name of the file to check.
' Outputs:  A boolean indicating whether or not the file exists.
' Assumes:  A fully qualified filename and path will be passed to
'           strFileName OR strFileName in in the current directory.
'**************************************************************************
Public Function FileExists(strFileName As String) As Boolean
    On Error Resume Next
    FileExists = IIf(FileLen(strFileName), True, False)
End Function
'**************************************************************************
' Purpose:  Creates a binary file for use with the other methods and
'           properties in this class.
' Inputs:   A new filename.
' Outputs:  Success or failure of this method.
' Effects:  Creates a disk file.
' Assumes:  A fully qualified filename (WITH PATH) is provided to
'           strFileName.
'**************************************************************************
Public Function CreateFile(strFileName As String) As Boolean
    '---------------------------------------------------------------------
    ' Set up an error handler that will exit the function immediately on
    ' any error.
    '---------------------------------------------------------------------
    On Error GoTo CreateFile_Err
    '---------------------------------------------------------------------
    ' Create the file.
    Open strFileName For Binary Access Read Write Lock Read Write As mintFileHandle
    '---------------------------------------------------------------------
    ' Since the file was sucessfully opened, then update the class-level
    ' variables. Also notify the caller that OpenFile succeded.
    '---------------------------------------------------------------------
    mstrFileName = strFileName
    mblnReadOnly = False
    mblnFileOpen = True
    CreateFile = True
    Exit Function
'-------------------------------------------------------------------------
' If an error occurs, then notify the user of the error number and message
' and exit the function.
'-------------------------------------------------------------------------
CreateFile_Err:
    MsgBox "Error " & Err.Number & " - " & Err.Description, vbCritical
    Exit Function
End Function
'**************************************************************************
' Purpose:  Opens a binary file for use with the other methods and
'           properties in this class.
' Inputs:   A filename, and an option to open the file as read-only.
' Outputs:  Success or failure of this method.
' Effects:  Opens a disk file.
' Assumes:  A fully qualified filename (WITH PATH) is provided to
'           strFileName.
'**************************************************************************
Public Function OpenFile(strFileName As String, _
                                        blnReadOnly As Boolean) As Boolean
    '---------------------------------------------------------------------
    ' If the file doesn't exist, then stop immediately.
    '---------------------------------------------------------------------
    If FileExists(strFileName) = False Then Exit Function
    '---------------------------------------------------------------------
    ' Set up an error handler that will exit the function immediately on
    ' any error.
    '---------------------------------------------------------------------
    On Error GoTo OpenFile_Err
    '---------------------------------------------------------------------
    ' Open the file the way the user requested.
    '
    ' NOTE
    '       Read-Only files will be shared, but Read/Write files will be
    '       Writelocked to prevent other users from opening the file and
    '       making changes which will be lost when this file is closed.
    '---------------------------------------------------------------------
    If blnReadOnly Then
        Open strFileName For Binary Access Read Shared As mintFileHandle
    Else
        Open strFileName For Binary Access Read Write Lock Write As mintFileHandle
    End If
    '---------------------------------------------------------------------
    ' Since the file was sucessfully opened, then update the class-level
    ' variables. Also notify the caller that OpenFile succeded.
    '---------------------------------------------------------------------
    mstrFileName = strFileName
    mblnReadOnly = blnReadOnly
    mblnFileOpen = True
    OpenFile = True
    Exit Function
'-------------------------------------------------------------------------
' If an error occurs, then notify the user of the error number and message
' and exit the function.
'-------------------------------------------------------------------------
OpenFile_Err:
    MsgBox "Error " & Err.Number & " - " & Err.Description, vbCritical
    Exit Function
End Function
'**************************************************************************
' Purpose:  Reads binary data from a file.
' Inputs:   The caller has the option to spefcify how many characters to
'           read from the file. If this option isn't specified, then the
'           entire file will be read.
' Outputs:  The data read from the file.
' Effects:  The file pointer will be moved.
' Assumes:  The value passed to vntChars will be valid.
'**************************************************************************
Public Function Read(Optional vntChars) As Variant
    Dim dblChars As Double  ' Holds the number of characters to read.
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    '---------------------------------------------------------------------
    ' Determine how many characters to read.
    '---------------------------------------------------------------------
    dblChars = IIf(IsMissing(vntChars), LOF(mintFileHandle), vntChars)
    '---------------------------------------------------------------------
    ' If vntChars wasn't provided, then the entire file will be read. This
    ' means the we must start at the top of the file.
    '---------------------------------------------------------------------
    If IsMissing(vntChars) Then Seek mintFileHandle, 1
    '---------------------------------------------------------------------
    ' Grab the entire chunk at once.
    '---------------------------------------------------------------------
    Read = Input(dblChars, mintFileHandle)
End Function
'**************************************************************************
' Purpose:  Reads a file into a string array.
' Inputs:   A pointer to a variable-length array, and the number of
'           characters to read.
' Outputs:  The number of elements in the array.
' Assumes:  The value passed to vntChars will be valid, and the caller
'           will handle changing the MousePointer.
'**************************************************************************
Public Function ReadFileIntoArray(strArray() As String, Optional vntChars) As Double
    Dim intCounter As Integer           ' Loop counter.
    Dim intNumChunks As Integer         ' Holds the upper index of the array.
    Dim dblFirstChunk As Double         ' Size of the first chunk to read.
    Dim dblChars As Double              ' Number of characters to read
                                        ' during each iteration of the loop.
    Dim dblOriginalPostion As Double    ' Holds the current file pointer.
    
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    '---------------------------------------------------------------------
    ' If vntChars wasn't provided, then use the length of the file.
    '---------------------------------------------------------------------
    dblChars = IIf(IsMissing(vntChars), LOF(mintFileHandle), CDbl(vntChars))
    '---------------------------------------------------------------------
    ' Remember the current file pointer before moving it.
    '---------------------------------------------------------------------
    dblOriginalPostion = Seek(mintFileHandle)
    If IsMissing(vntChars) Then Seek mintFileHandle, 1
    '---------------------------------------------------------------------
    ' If there is only one chunk, then read it and abort.
    '---------------------------------------------------------------------
    If dblChars < TextBoxLimit Then
        ReDim strArray(1)
        strArray(0) = Input(dblChars, mintFileHandle)
        Seek mintFileHandle, dblOriginalPostion
        ReadFileIntoArray = 1
        Exit Function
    End If
    '---------------------------------------------------------------------
    ' Find out how many even chunks to read, and save the remainder for
    ' the first chunk.
    '---------------------------------------------------------------------
    dblFirstChunk = dblChars Mod TextBoxLimit
    intNumChunks = dblChars \ TextBoxLimit
    '---------------------------------------------------------------------
    ' Resize the array, destroying any existing data in it.
    '---------------------------------------------------------------------
    ReDim strArray(intNumChunks + IIf(dblFirstChunk, 1, 0))
    '---------------------------------------------------------------------
    ' If there was a remainder stored in dblFirstChunk, then read that
    ' chunk.
    '---------------------------------------------------------------------
    If dblFirstChunk Then
        strArray(0) = Input(dblFirstChunk, mintFileHandle)
        intCounter = 1
    End If
    '---------------------------------------------------------------------
    ' Cache this value to speed up the loop on huge files.
    '---------------------------------------------------------------------
    intNumChunks = UBound(strArray) - 1
    '---------------------------------------------------------------------
    ' Load the array using TextBoxLimit size chunks.
    '---------------------------------------------------------------------
    For intCounter = intCounter To intNumChunks
        strArray(intCounter) = Input(TextBoxLimit, mintFileHandle)
    Next intCounter
    '---------------------------------------------------------------------
    ' Restore the file cursor position to its original state.
    '---------------------------------------------------------------------
    Seek mintFileHandle, dblOriginalPostion
    '---------------------------------------------------------------------
    ' Return the number of elements in the new array.
    '---------------------------------------------------------------------
    ReadFileIntoArray = intNumChunks + 1
End Function
'**************************************************************************
' Purpose:  Writes data to a binary file.
' Inputs:   Any type of data, an optional byte postion to write the data,
'           and an option to write append a carriage-return at the end
'           of the data written to the file.
' Effects:  Writes data to a file.
' Assumes:  Optional arguments contain valid values.
'**************************************************************************
Public Sub WriteFile(ByVal vntWrite As Variant, Optional vntBytePos, _
                                                Optional vntAppendReturn)
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    '---------------------------------------------------------------------
    ' If a value for vntAppendReturn wasn't provide, then append a
    ' carriage-return line feed to the data to be written to the file. If
    ' ANY value was provided, then ignore this option.
    '---------------------------------------------------------------------
    If IsMissing(vntAppendReturn) Then vntWrite = vntWrite & vbCrLf
    '---------------------------------------------------------------------
    ' If no byte postion was provided, then write at the current position.
    '---------------------------------------------------------------------
    If IsMissing(vntBytePos) Then
        Put mintFileHandle, , vntWrite
    '---------------------------------------------------------------------
    ' Otherwise, write at the position provided or the end of file.
    '---------------------------------------------------------------------
    Else
        If (vntBytePos < 1) Or (vntBytePos > (LOF(mintFileHandle) - 1)) Then
            vntBytePos = LOF(mintFileHandle) - 1
        End If
        Put mintFileHandle, vntBytePos, vntWrite
    End If
End Sub
'**************************************************************************
' Purpose:  Writes a string to a binary file.
' Inputs:   Any type of data, an optional byte postion to write the data,
'           and an option to write append a carriage-return at the end
'           of the data written to the file.
' Effects:  Writes a string to a file.
' Assumes:  Optional arguments contain valid values.
'**************************************************************************
Public Sub WriteString(ByVal strWrite As String, Optional vntBytePos, _
                                                Optional vntAppendReturn)
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    '---------------------------------------------------------------------
    ' If a value for vntAppendReturn wasn't provide, then append a
    ' carriage-return line feed to the data to be written to the file. If
    ' ANY value was provided, then ignore this option.
    '---------------------------------------------------------------------
    If IsMissing(vntAppendReturn) Then strWrite = strWrite & vbCrLf
    '---------------------------------------------------------------------
    ' If no byte postion was provided, then write at the current position.
    '---------------------------------------------------------------------
    If IsMissing(vntBytePos) Then
        Put mintFileHandle, , strWrite
    '---------------------------------------------------------------------
    ' Otherwise, write at the position provided or the end of file.
    '---------------------------------------------------------------------
    Else
        If (vntBytePos < 1) Or (vntBytePos > (LOF(mintFileHandle) - 1)) Then
            vntBytePos = LOF(mintFileHandle) - 1
        End If
        Put mintFileHandle, vntBytePos, strWrite
    End If
End Sub
'**************************************************************************
' Purpose:  Writes an string array of data to a file.
' Inputs:   A pointer to a loaded array, and the same optional
'           arguments provided by WriteFile
' Effects:  Changes the disk file.
' Assumes:  strArray is an array that has been loaded with data.
'**************************************************************************
Public Sub WriteArrayToFile(strArray() As String, Optional vntBytePos, Optional vntAppendReturn)
    Dim intNumElements As Integer   ' Number of elements in the array.
    Dim intCounter As Integer       ' Loop counter.
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    '---------------------------------------------------------------------
    ' Cache upper index of the array.
    '---------------------------------------------------------------------
    intNumElements = UBound(strArray)
    '---------------------------------------------------------------------
    ' Loop through the array, calling WriteFile with each element.
    '---------------------------------------------------------------------
    For intCounter = 0 To intNumElements
        WriteString strArray(intCounter), vntBytePos, vntAppendReturn
    Next intCounter
End Sub
'**************************************************************************
' Purpose:  Loads a TextBox control with text from a binary file.
' Inputs:   A pointer to a TextBox control.
' Effects:  Replaces the contents of a TextBox control's .Text property
'           with the text in the current file.
' Notes:    Only a maximum of TextBoxLimit will be read into the control.
'**************************************************************************
Public Sub LoadTextBox(txtBox As TextBox)
    Dim dblCurPos As Double     ' Holds the file pointer position.
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    '---------------------------------------------------------------------
    ' Save the file pointer, read the data into the text box, then restore
    ' the file pointer.
    '---------------------------------------------------------------------
    dblCurPos = Seek(mintFileHandle)
    Seek mintFileHandle, 1
    If LOF(mintFileHandle) > TextBoxLimit Then
        txtBox.Text = Read(TextBoxLimit)
    Else
        txtBox.Text = Read
    End If
    Seek mintFileHandle, dblCurPos
End Sub
'**************************************************************************
' Purpose:  Dumps the currently open file to the printer.
' Outputs:  A printed document.
' Effects:  Temporarily loads the entire file into memory.
' Assumes:  A valid printer has been selected, and there is enough memory
'           to load the entire file.
'**************************************************************************
Public Sub PrintFile()
    Dim strArray() As String        ' Used to hold the contents of the file.
    Dim intCounter As Integer       ' Loop counter.
    Dim intNumElements As Integer   ' The number of elements in the array.
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    '---------------------------------------------------------------------
    ' Read the file into the array.
    '---------------------------------------------------------------------
    intNumElements = ReadFileIntoArray(strArray) - 1
    '---------------------------------------------------------------------
    ' Send each element of the array to the printer.
    '---------------------------------------------------------------------
    For intCounter = 0 To intNumElements
        Printer.Print strArray(intCounter)
    Next intCounter
    '---------------------------------------------------------------------
    ' Complete the print job.
    '---------------------------------------------------------------------
    Printer.EndDoc
End Sub
'**************************************************************************
' Purpose:  Moves the file pointer to the end of file.
' Effects:  The next write will destroy the EOF pointer in the file, unless
'           cursor position is moved back 1 byte.
'**************************************************************************
Public Sub MoveEOF()
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    Seek mintFileHandle, LOF(mintFileHandle)
End Sub
'**************************************************************************
' Purpose:  Moves the file pointer to the top of the file.
'**************************************************************************
Public Sub MoveBOF()
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    Seek mintFileHandle, 1
End Sub
'**************************************************************************
' Purpose:  Retrives the path where the currently open file resides.
' Effects:  The value returned includes the trailing backslash.
'**************************************************************************
Public Property Get FilePath()
    FilePath = GetFileInfo(mstrFileName, True)
End Property
'**************************************************************************
' Purpose:  Retrives the strFileName argument passed to OpenFile.
'**************************************************************************
Public Property Get FullFilePath()
    FullFilePath = mstrFileName
End Property
'**************************************************************************
' Purpose:  Returns the extension of the current file.
' Assumes:  The current file has an extension.
'**************************************************************************
Public Property Get Extension()
    Dim strExtension As String  ' Temporary local variable.
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    '---------------------------------------------------------------------
    ' Search the last 4 characters for a period to determine the extension
    '---------------------------------------------------------------------
    strExtension = Right(mstrFileName, 4)
    Extension = Mid(strExtension, InStr(strExtension, ".") + 1)
End Property
'**************************************************************************
' Purpose:  Returns a boolean result of any specific file attribute.
' Inputs:   Any of the constants listed in help topic for GetAttr.
' Outputs:  A boolean result of the attribute state.
'**************************************************************************
Public Function GetAttribute(intAttributeMask) As Boolean
    GetAttribute = GetAttr(mstrFileName) And intAttributeMask
End Function
'**************************************************************************
' Purpose:  To allow the caller to force the file to be closed.
' Effects:  Commits all file modifications to the hard disk.
'**************************************************************************
Public Sub CloseFile()
    Close mintFileHandle
    mblnFileOpen = False
End Sub
'**************************************************************************
' Purpose:  Clears the file to zero bytes, without removing it from the
'           hard disk.
' Effects:  A zero byte file remains on the hard disk.
'**************************************************************************
Public Sub Clear()
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    '---------------------------------------------------------------------
    ' Close the file, open it for output, and re-close it.
    '---------------------------------------------------------------------
    Close mintFileHandle
    Open mstrFileName For Output As mintFileHandle
    Close mintFileHandle
    '---------------------------------------------------------------------
    ' Now that the file is clear, reopen it.
    '---------------------------------------------------------------------
    OpenFile mstrFileName, mblnReadOnly
End Sub
'**************************************************************************
' Purpose:  Moves the file pointer to a specific position within the file.
'**************************************************************************
Public Sub FileSeek(dblFilePostion As Double)
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    
    Seek mintFileHandle, dblFilePostion
End Sub
'**************************************************************************
' Purpose:  Retrieves a single byte from a file.
' Inputs:   The caller may optionally specify where to begin reading.
'**************************************************************************
Public Function GetByte(Optional vntFilePosition) As Byte
    Dim bytBuffer As Byte   ' Used to hold the data read from the file.
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    If IsMissing(vntFilePosition) Then
        Get mintFileHandle, , bytBuffer
    Else
        Get mintFileHandle, vntFilePosition, bytBuffer
    End If
    GetByte = bytBuffer
End Function
'**************************************************************************
' Purpose:  Writes a single byte to a file.
' Inputs:   The caller may optionally specify where to begin writing.
'**************************************************************************
Public Sub PutByte(bytData As Byte, Optional vntFilePosition)
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    If IsMissing(vntFilePosition) Then
        Put mintFileHandle, , bytData
    Else
        Put mintFileHandle, vntFilePosition, bytData
    End If
End Sub
'**************************************************************************
' Purpose:  Breaks strFileName into path and filename parts, and returns
'           the desired part specified by blnReturnPath.
' Inputs:   A filename and a boolean flag.
' Outputs:
' Effects:
' Assumes:
'**************************************************************************
Private Function GetFileInfo(ByVal strFileName As String, blnReturnPath _
                                                    As Boolean) As String
    Dim intLastSlash As Integer     ' Location of the last slash found.
    Dim intWhere As Integer         ' Result of call to InStr.
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    '---------------------------------------------------------------------
    ' Loop until there are no more slashes.
    '---------------------------------------------------------------------
    Do
        intLastSlash = intWhere
        intWhere = InStr(intLastSlash + 1, strFileName, "\")
    Loop While (intWhere)
    '---------------------------------------------------------------------
    ' Return the requested result. Path is to the left and filename is
    ' to the right.
    '---------------------------------------------------------------------
    If blnReturnPath Then
        GetFileInfo = Left(strFileName, intLastSlash)
    Else
        GetFileInfo = Mid(strFileName, intLastSlash + 1)
    End If
End Function
'**************************************************************************
' Purpose:  Removes the file from the hard drive.
' Assumes:  You have write access to the disk where the file exists.
'**************************************************************************
Public Sub Delete()
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    Close mintFileHandle
    Kill mstrFileName
    mblnFileOpen = False
End Sub
'**************************************************************************
' Purpose:  Generic error message for attempts to call methods or
'           properties which require OpenFile to be called successfully.
'**************************************************************************
Private Sub Err_FileNotOpen()
    MsgBox "You must call OpenFile before you may call this method or property.", vbCritical
End Sub
'**************************************************************************
' Purpose:  Returns the byte location of a word within a file.
' Inputs:   A search word, a case-senstivity flag, and a postion where
'           search should begin.
' Outputs:  A byte location within the file.
' Effects:  Loads the entire file into a string array.
' Assumes:  Available memory and the user will maintain the MousePointer.
'**************************************************************************
Public Function Find(strFindWhat As String, blnCaseSensitive As Boolean, _
                        Optional vntStartPosition) As Double
    Dim strFile() As String         ' Holds the file.
    Dim intCounter As Integer       ' Loop counter.
    Dim intIterations As Integer    ' Cache variable.
    Dim intWasFound As Integer      ' Location where the word was found.
    Dim dblResult As Double         ' Holds the total chars searched.
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    '---------------------------------------------------------------------
    ' Read the file into an array, and cache the UBound index.
    '---------------------------------------------------------------------
    intIterations = ReadFileIntoArray(strFile) - 1
    '---------------------------------------------------------------------
    ' Loop through the file and call FindInString with each element of the
    ' array.
    '---------------------------------------------------------------------
    For intCounter = 0 To intIterations
        intWasFound = FindInString(strFile(intCounter), strFindWhat, _
                                    blnCaseSensitive, vntStartPosition)
        '-----------------------------------------------------------------
        ' Stop on first find and ignore the last line in the loop body.
        '-----------------------------------------------------------------
        If intWasFound Then Exit For
        dblResult = dblResult + Len(strFile(intCounter))
    Next intCounter
    '---------------------------------------------------------------------
    ' Return the running total (if any) plus the word location.
    '---------------------------------------------------------------------
    Find = dblResult + intWasFound
End Function
'**************************************************************************
' Purpose:  Helper function that searches a single string.
' Inputs:   See above + a source string.
' Outputs:  See above.
'**************************************************************************
Private Function FindInString(strSource As String, strFindWhat As String, _
            blnCaseSensitive As Boolean, Optional vntStartPos) As Integer
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    '---------------------------------------------------------------------
    ' Provide a default value.
    '---------------------------------------------------------------------
    If IsMissing(vntStartPos) Then vntStartPos = 1
    '---------------------------------------------------------------------
    ' Let InStr do the work.
    '---------------------------------------------------------------------
    FindInString = InStr(vntStartPos, strSource, strFindWhat, blnCaseSensitive)
End Function
'**************************************************************************
' Purpose:  Replaces every occurance of a given word with another in the
'           entire file.
' Inputs:   Search and Replace strings and a case-sensitivity flag.
' Outputs:  The number of replacements.
' Effects:  The file is changed permanently.
' Assumes:  See Find.
'**************************************************************************
Public Function ReplaceAll(strFindWhat As String, strReplaceWith As String, _
                                    blnCaseSensitive As Boolean) As Integer
    Dim strFile() As String         ' The source file.
    Dim intCounter As Integer       ' Loop counter.
    Dim intIterations As Integer    ' UBound index.
    Dim intReplacements As Integer  ' Number of replacements.
    Dim dblCurPos As Double         ' Current cursor position.
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    '---------------------------------------------------------------------
    ' Load the file and cache the UBound.
    '---------------------------------------------------------------------
    intIterations = ReadFileIntoArray(strFile) - 1
    '---------------------------------------------------------------------
    ' Loop through the file and call ReplaceAllInString to do the replace.
    '---------------------------------------------------------------------
    For intCounter = 0 To intIterations
        intReplacements = intReplacements + _
                ReplaceAllInString(strFile(intCounter), strFindWhat, _
                                        strReplaceWith, blnCaseSensitive)
    Next intCounter
    '---------------------------------------------------------------------
    ' If any replacements were made, then replace the disk file and
    ' reopen it.
    '---------------------------------------------------------------------
    If intReplacements Then
        dblCurPos = Seek(mintFileHandle)
        Close mintFileHandle
        Kill mstrFileName
        CreateFile mstrFileName
        WriteArrayToFile strFile, vntAppendReturn:=False
        Seek mintFileHandle, dblCurPos
    End If
    
    ReplaceAll = intReplacements
End Function
'**************************************************************************
' Purpose:  Helper function that performs a ReplaceAll within a string.
' Inputs:   See above + a source string.
' Outputs:  See above.
' Effects:  See above.
' Assumes:  See above.
'**************************************************************************
Private Function ReplaceAllInString(strSource As String, strFindWhat As _
                                    String, strReplaceWith As String, _
                                    blnCaseSensitive As Boolean) As Integer
    Dim intWhere As Integer             ' Location of strFindWhat.
    Dim strReplacedString As String     ' Result string.
    Dim intNumChanges As Integer        ' Number of changes made.
    Dim intFindWhatLen As Integer       ' Len of strFindWhat.
    '---------------------------------------------------------------------
    ' If the OpenFile hasn't been called sucessfully, then abort.
    '---------------------------------------------------------------------
    If Not mblnFileOpen Then Err_FileNotOpen
    '---------------------------------------------------------------------
    ' Cache this value.
    '---------------------------------------------------------------------
    intFindWhatLen = Len(strFindWhat)
    '---------------------------------------------------------------------
    ' Do an initial search.
    '---------------------------------------------------------------------
    intWhere = InStr(1, strSource, strFindWhat, blnCaseSensitive)
    '---------------------------------------------------------------------
    ' If found, then begin loop, increment and search again.
    '---------------------------------------------------------------------
    Do While intWhere
        intNumChanges = intNumChanges + 1
        strReplacedString = strReplacedString & Left(strSource, intWhere - 1) _
                            & strReplaceWith
        strSource = Mid(strSource, intWhere + intFindWhatLen)
        intWhere = InStr(1, strSource, strFindWhat, blnCaseSensitive)
    Loop
    '---------------------------------------------------------------------
    ' Complete modifing strSource, and return the result.
    '---------------------------------------------------------------------
    If intNumChanges Then
        strSource = strReplacedString & strSource
    End If
    
    ReplaceAllInString = intNumChanges

End Function
