package lib::FileDB;
$VERSION = 1.00;
use CGI::Carp qw(fatalsToBrowser);
use lib::MsgBox;
use strict;

{
# Encapsulated class data

	my @_table_records;
	my $sort_index;

# Class methods, to operate on encapsulated class data

	# Reset records
	sub _reset_records
	{
		@_table_records = ();
	}

	# Read all records into array
	sub _read_records
	{
		@_table_records = <DB>;
		chomp(@_table_records);
	}

	# Return all stored records
	sub _get_records
	{
		return @_table_records;
	}

	# Update records
	sub _update_records
	{
		my ($self, @updated_records) = @_;

		# Make sure that field names are intact
		if ($_table_records[0] eq $updated_records[0]) {
			@_table_records = ();
		} elsif ($_table_records[0] ne "") {
			@_table_records = ($_table_records[0]);
		}

		push(@_table_records, @updated_records);
	}

	# Given field name and array of fields,
	# return field index in the array
	sub _get_field_index
	{
		my ($self, $field_name, @fields) = @_;

		my ($i, $found) = (0, 0);
		for ($i = 0; $i < scalar(@fields); $i++) {
			if ($fields[$i] eq $field_name || lc($fields[$i]) eq $field_name) {
				$found = 1;
				last;
			}
		}

		croak "Field name $field_name was not found." if !$found;

		return $i;
	}
	
	# Sort records
	sub _sort_records
	{
		my ($self, $sort_field_index, $sort_type, $sort_order, @unsorted_records) = @_;

		my @sorted_records;

		$sort_index = $sort_field_index;

		if ($sort_type eq "numeric") {
			if ($sort_order eq "ascending") {
				@sorted_records = sort _numeric_ascending @unsorted_records;
			}
			elsif ($sort_order eq "descending") {
				@sorted_records = sort _numeric_descending @unsorted_records;
			}
		}
		elsif ($sort_type eq "alphabetic") {
			if ($sort_order eq "ascending") {
				@sorted_records = sort _alphabetic_ascending @unsorted_records;
			}
			elsif ($sort_order eq "descending") {
				@sorted_records = sort _sort_alphabetic_descending @unsorted_records;
			}
		}
		
		return @sorted_records;
	}

	# Numeric ascending sort
	sub _numeric_ascending
	{
		my @a = split "\t", $a;
		my @b = split "\t", $b;
		return ($a[$sort_index] <=> $b[$sort_index] || $a[$sort_index] cmp $b[$sort_index]);
	}

	# Numeric descending sort
	sub _numeric_descending
	{
		my @a = split "\t", $a;
		my @b = split "\t", $b;
		return ($b[$sort_index] <=> $a[$sort_index] || $a[$sort_index] cmp $b[$sort_index]);
	}

	# Alphabetic ascending sort
	sub _alphabetic_ascending
	{
		my @a = split "\t", $a;
		my @b = split "\t", $b;
		return ($a[$sort_index] cmp $b[$sort_index] || $a[$sort_index] <=> $b[$sort_index]);
	}

	# Alphabetic descending sort
	sub _sort_alphabetic_descending
	{
		my @a = split "\t", $a;
		my @b = split "\t", $b;
		return ($b[$sort_index] cmp $a[$sort_index] || $b[$sort_index] <=> $a[$sort_index]);
	}
}

# Constructor
sub new
{
	my ($class, %arg) = @_;
	my $self = bless {}, $class;
}

# Create file
sub create_file
{
	my ($self, %arg) = @_;

	$self->open_file( -file_name => $arg{'-file_name'}, 
	                  -open_to   => "overwrite"
	                );

	$self->close_file();
}

# Open file
sub open_file
{
	my ($self, %arg) = @_;

	# Error checking
	if (!$arg{'-file_name'}) {
		croak "Incorrect use of lib::FileDB::open_file() method: missing or invalid -file_name parameter.";
	}

	# Defaults
	$arg{'-open_to'} = "read" if !$arg{'-open_to'};

	# Reset record container
	$self->_reset_records();

	# Open file
	if ($arg{'-open_to'} eq "read") {
		open(DB,"$arg{'-file_name'}") || lib::MsgBox->new(text => "Could not open $arg{'-file_name'}: $!");
		$self->_read_records();
	}
	elsif ($arg{'-open_to'} eq "update") {
		open(DB,"+<$arg{'-file_name'}") || lib::MsgBox->new(text => "Could not update $arg{'-file_name'}: $!");
		flock(DB,2);
		$self->_read_records();
		seek(DB,0,0);
		truncate(DB,0);
	}
	elsif ($arg{'-open_to'} eq "append") {
		open(DB,">>$arg{'-file_name'}") || lib::MsgBox->new(text => "Could not update $arg{'-file_name'}: $!");
		flock(DB,2);
		seek(DB,0,2);
	}
	elsif ($arg{'-open_to'} eq "overwrite") {
		open(DB,">$arg{'-file_name'}") || lib::MsgBox->new(text => "Could not create $arg{'-file_name'}: $!");
		flock(DB,2);
		seek(DB,0,0);
	}
	else {
		croak "Incorrect use of lib::FileDB::open_file() method: -open_to parameter must be set to 'read', 'update', 'append', or 'overwrite'.";
	}
}

# Close file
sub close_file
{
	close(DB);
}

# Given condition, find and return table record
sub get_record
{
	my ($self, %arg) = @_;

	# Error checking
	if (!$arg{'-condition'}) {
		croak "Incorrect use of lib::FileDB::get_record() method: missing or invalid -condition parameter.";
	}

	# Decode condition
	my ($cond_field, $cond_value) = split "=", $arg{'-condition'};

	# Get table fields and records
	my @records = $self->_get_records();
	my $field_names = shift(@records);
	my @fields = split "\t", $field_names;

	# Build a search string
	my $search_string = "^";
	for (my $i = 0; $i < scalar(@fields); $i++) {
		if ($fields[$i] eq $cond_field || lc($fields[$i]) eq $cond_field) {
			$search_string .= $cond_value;
		} else {
			$search_string .= '[\S ]*';
		}
		if ($i != $#fields) {
			$search_string .= "\t";
		}
	}

	# Perform search
	my @match = grep(/$search_string/i, @records);

	# Write record to hash
	my ($lc_field, %rec_info);
	if (@match) {
		my @matching_fields = split "\t", $match[0], scalar(@fields);
		for (my $j = 0; $j < scalar(@fields); $j++) {
			$lc_field = lc($fields[$j]);
			$rec_info{$fields[$j]} = $matching_fields[$j];
			$rec_info{$lc_field}   = $matching_fields[$j];
		}
	}

	return %rec_info;
}

# Given condition (optional), find and return all matching table records
sub get_records
{
	my ($self, %arg) = @_;

	# Defaults
	if (!$arg{'-sort_type'}) {
		$arg{'-sort_type'} = "numeric";
	}
	if (!$arg{'-sort_order'}) {
		if ($arg{'-sort_type'} eq "numeric") {
			$arg{'-sort_order'} = "descending";
		}
		elsif ($arg{'-sort_type'} eq "alphabetic") {
			$arg{'-sort_order'} = "ascending";
		}
	}

	# Get table fields and records
	my @records = $self->_get_records();
	my $field_names = shift(@records);
	my @fields = split "\t", $field_names;

	# Process query
	my ($cond_field, $cond_value, @matches);
	if (!$arg{'-condition'}) {
		@matches = @records;
	}
	elsif ($arg{'-condition'} =~ /^\w+>=/) {

		($cond_field, $cond_value) = split ">=", $arg{'-condition'};

		# Determine the index of condition field
		my $field_index = $self->_get_field_index($cond_field, @fields);

		# Find matching records
		my @rec_fields;
		foreach my $rec(@records) {
			@rec_fields = split "\t", $rec, scalar(@fields);
			if ($rec_fields[$field_index] >= $cond_value) {
				push(@matches, $rec);
			}
		}
	}
	elsif ($arg{'-condition'} =~ /^\w+>/) {

		($cond_field, $cond_value) = split ">", $arg{'-condition'};

		# Determine the index of condition field
		my $field_index = $self->_get_field_index($cond_field, @fields);

		# Find matching records
		my @rec_fields;
		foreach my $rec(@records) {
			@rec_fields = split "\t", $rec, scalar(@fields);
			if ($rec_fields[$field_index] > $cond_value) {
				push(@matches, $rec);
			}
		}
	}
	elsif ($arg{'-condition'} =~ /^\w+<=/) {

		($cond_field, $cond_value) = split "<=", $arg{'-condition'};

		# Determine the index of condition field
		my $field_index = $self->_get_field_index($cond_field, @fields);

		# Find matching records
		my @rec_fields;
		foreach my $rec(@records) {
			@rec_fields = split "\t", $rec, scalar(@fields);
			if ($rec_fields[$field_index] <= $cond_value) {
				push(@matches, $rec);
			}
		}
	}
	elsif ($arg{'-condition'} =~ /^\w+</) {

		($cond_field, $cond_value) = split "<", $arg{'-condition'};

		# Determine the index of condition field
		my $field_index = $self->_get_field_index($cond_field, @fields);

		# Find matching records
		my @rec_fields;
		foreach my $rec(@records) {
			@rec_fields = split "\t", $rec, scalar(@fields);
			if ($rec_fields[$field_index] < $cond_value) {
				push(@matches, $rec);
			}
		}
	}
	elsif ($arg{'-condition'} =~ /^\w+!=/) {

		($cond_field, $cond_value) = split "!=", $arg{'-condition'};

		# Determine the index of condition field
		my $field_index = $self->_get_field_index($cond_field, @fields);

		# Find matching records
		my @rec_fields;
		foreach my $rec(@records) {
			@rec_fields = split "\t", $rec, scalar(@fields);
			if ($rec_fields[$field_index] ne $cond_value) {
				push(@matches, $rec);
			}
		}
	}
	elsif ($arg{'-condition'} =~ /^\w+=/) {

		($cond_field, $cond_value) = split "=", $arg{'-condition'};

		# Build a search string
		my $search_string = "^";
		for (my $i = 0; $i < scalar(@fields); $i++) {
			if ($fields[$i] eq $cond_field || lc($fields[$i]) eq $cond_field) {
				$search_string .= $cond_value;
			} else {
				$search_string .= '[\S ]*';
			}
			if ($i != $#fields) {
				$search_string .= "\t";
			}
		}

		# Perform search
		@matches = grep(/$search_string/i, @records);
	}

	# Write records to hash
	my (%rec_info, $lc_field, @matching_fields);
	if (@matches) {

		# Sort records
		if ($arg{'-sort_by'}) {
			my $sort_field_index = $self->_get_field_index($arg{'-sort_by'}, @fields);
			@matches = $self->_sort_records($sort_field_index, $arg{'-sort_type'}, $arg{'-sort_order'}, @matches);
		}

		for (my $p = 0; $p < scalar(@matches); $p++) {
			@matching_fields = split "\t", $matches[$p], scalar(@fields);
			for (my $q = 0; $q < scalar(@fields); $q++) {
				$lc_field = lc($fields[$q]);
				$rec_info{$fields[$q]}[$p] = $matching_fields[$q];
				$rec_info{$lc_field}[$p]   = $matching_fields[$q];
			}
		}
	}

	return %rec_info;
}

# Update records
sub update_records
{
	my ($self, %arg) = @_;

	# Error checking
	if (!$arg{'-condition'}) {
		croak "Incorrect use of lib::FileDB::update_record() method: missing or invalid -condition parameter.";
	}
	if (!$arg{'-fields'}) {
		croak "Incorrect use of lib::FileDB::update_record() method: missing or invalid -fields parameter.";
	}
	if (!$arg{'-values'}) {
		croak "Incorrect use of lib::FileDB::update_record() method: missing or invalid -values parameter.";
	}

	if (scalar(@{$arg{'-fields'}}) != scalar(@{$arg{'-values'}})) {
		croak "Incorrect use of lib::FileDB::update_record() method: number of fields and values must be the same.";
	}

	# Get all records and fields
	my @records = $self->_get_records();
	my $field_names = shift(@records);
	my @fields = split "\t", $field_names;

	# Decode condition
	my ($cond_field, $cond_value);

	if ($arg{'-condition'} =~ /^\w+>=/) {

		($cond_field, $cond_value) = split ">=", $arg{'-condition'};

		# Determine the index of condition field
		my $field_index = $self->_get_field_index($cond_field, @fields);

		# Modify matching records
		my (@rec_fields, $updated_field_index);
		foreach my $rec(@records) {
			@rec_fields = split "\t", $rec, scalar(@fields);
			if ($rec_fields[$field_index] >= $cond_value) {
				for (my $i = 0; $i < scalar(@{$arg{'-fields'}}); $i++) {
					$updated_field_index = $self->_get_field_index($arg{'-fields'}[$i], @fields);
					$rec_fields[$updated_field_index] = $arg{'-values'}[$i];
				}
				$rec = join "\t", @rec_fields;
			}
		}
	}
	elsif ($arg{'-condition'} =~ /^\w+>/) {

		($cond_field, $cond_value) = split ">", $arg{'-condition'};

		# Determine the index of condition field
		my $field_index = $self->_get_field_index($cond_field, @fields);

		# Modify matching records
		my (@rec_fields, $updated_field_index);
		foreach my $rec(@records) {
			@rec_fields = split "\t", $rec, scalar(@fields);
			if ($rec_fields[$field_index] > $cond_value) {
				for (my $i = 0; $i < scalar(@{$arg{'-fields'}}); $i++) {
					$updated_field_index = $self->_get_field_index($arg{'-fields'}[$i], @fields);
					$rec_fields[$updated_field_index] = $arg{'-values'}[$i];
				}
				$rec = join "\t", @rec_fields;
			}
		}
	}
	elsif ($arg{'-condition'} =~ /^\w+<=/) {

		($cond_field, $cond_value) = split "<=", $arg{'-condition'};

		# Determine the index of condition field
		my $field_index = $self->_get_field_index($cond_field, @fields);

		# Modify matching records
		my (@rec_fields, $updated_field_index);
		foreach my $rec(@records) {
			@rec_fields = split "\t", $rec, scalar(@fields);
			if ($rec_fields[$field_index] <= $cond_value) {
				for (my $i = 0; $i < scalar(@{$arg{'-fields'}}); $i++) {
					$updated_field_index = $self->_get_field_index($arg{'-fields'}[$i], @fields);
					$rec_fields[$updated_field_index] = $arg{'-values'}[$i];
				}
				$rec = join "\t", @rec_fields;
			}
		}
	}
	elsif ($arg{'-condition'} =~ /^\w+</) {

		($cond_field, $cond_value) = split "<", $arg{'-condition'};

		# Determine the index of condition field
		my $field_index = $self->_get_field_index($cond_field, @fields);

		# Modify matching records
		my (@rec_fields, $updated_field_index);
		foreach my $rec(@records) {
			@rec_fields = split "\t", $rec, scalar(@fields);
			if ($rec_fields[$field_index] < $cond_value) {
				for (my $i = 0; $i < scalar(@{$arg{'-fields'}}); $i++) {
					$updated_field_index = $self->_get_field_index($arg{'-fields'}[$i], @fields);
					$rec_fields[$updated_field_index] = $arg{'-values'}[$i];
				}
				$rec = join "\t", @rec_fields;
			}
		}
	}
	elsif ($arg{'-condition'} =~ /^\w+=/) {

		($cond_field, $cond_value) = split "=", $arg{'-condition'};

		# Determine the index of condition field
		my $field_index = $self->_get_field_index($cond_field, @fields);

		# Modify matching records
		my (@rec_fields, $updated_field_index);
		foreach my $rec(@records) {
			@rec_fields = split "\t", $rec, scalar(@fields);
			if (lc($rec_fields[$field_index]) eq lc($cond_value)) {
				for (my $i = 0; $i < scalar(@{$arg{'-fields'}}); $i++) {
					$updated_field_index = $self->_get_field_index($arg{'-fields'}[$i], @fields);
					$rec_fields[$updated_field_index] = $arg{'-values'}[$i];
				}
				$rec = join "\t", @rec_fields;
			}
		}
	}

	# Save modified table records
	$self->_update_records(@records);
}

# Insert new records
sub insert_record
{
	my ($self, %arg) = @_;

	# Error checking
	if (!$arg{'-values'}) {
		croak "Incorrect use of lib::FileDB::insert_record() method: missing or invalid -values parameter.";
	}

	# Get all records and fields
	my @records = $self->_get_records();
	my $field_names = shift(@records);
	my @fields = split "\t", $field_names, -1;	# LIMIT = -1 produces arbitrary large limit, includes empty pieces

	# All fields are required
	if (@fields && (scalar(@fields) != scalar(@{$arg{'-values'}}))) {
		croak @{$arg{'-values'}}, "Incorrect use of lib::FileDB::insert_record() method: you must supply values for every table field.";
	}

	my $new_record = join "\t", @{$arg{'-values'}};
	push(@records, $new_record);

	# Save modified table records
	$self->_update_records(@records);
}

# Delete records
sub delete_records
{
	my ($self, %arg) = @_;

	# Get all records and fields
	my @records = $self->_get_records();
	my $field_names = shift(@records);
	my @fields = split "\t", $field_names;

	# Decode condition
	my ($cond_field, $cond_value);


	if (!$arg{'-condition'}) {
		@records = ();
	}
	elsif ($arg{'-condition'} =~ /^\w+>=/) {

		($cond_field, $cond_value) = split ">=", $arg{'-condition'};

		# Determine the index of condition field
		my $field_index = $self->_get_field_index($cond_field, @fields);

		# Delete matching records
		my (@rec_fields, $updated_field_index);
		for (my $i = 0; $i < scalar(@records); $i++) {
			@rec_fields = split "\t", $records[$i], scalar(@fields);
			if ($rec_fields[$field_index] >= $cond_value) {
				splice(@records, $i--, 1);
			}
		}
	}
	elsif ($arg{'-condition'} =~ /^\w+>/) {

		($cond_field, $cond_value) = split ">", $arg{'-condition'};

		# Determine the index of condition field
		my $field_index = $self->_get_field_index($cond_field, @fields);

		# Delete matching records
		my (@rec_fields, $updated_field_index);
		for (my $i = 0; $i < scalar(@records); $i++) {
			@rec_fields = split "\t", $records[$i], scalar(@fields);
			if ($rec_fields[$field_index] > $cond_value) {
				splice(@records, $i--, 1);
			}
		}
	}
	elsif ($arg{'-condition'} =~ /^\w+<=/) {

		($cond_field, $cond_value) = split "<=", $arg{'-condition'};

		# Determine the index of condition field
		my $field_index = $self->_get_field_index($cond_field, @fields);

		# Delete matching records
		my (@rec_fields, $updated_field_index);
		for (my $i = 0; $i < scalar(@records); $i++) {
			@rec_fields = split "\t", $records[$i], scalar(@fields);
			if ($rec_fields[$field_index] <= $cond_value) {
				splice(@records, $i--, 1);
			}
		}
	}
	elsif ($arg{'-condition'} =~ /^\w+</) {

		($cond_field, $cond_value) = split "<", $arg{'-condition'};

		# Determine the index of condition field
		my $field_index = $self->_get_field_index($cond_field, @fields);

		# Delete matching records
		my (@rec_fields, $updated_field_index);
		for (my $i = 0; $i < scalar(@records); $i++) {
			@rec_fields = split "\t", $records[$i], scalar(@fields);
			if ($rec_fields[$field_index] < $cond_value) {
				splice(@records, $i--, 1);
			}
		}
	}
	elsif ($arg{'-condition'} =~ /^\w+=/) {

		($cond_field, $cond_value) = split "=", $arg{'-condition'};

		# Determine the index of condition field
		my $field_index = $self->_get_field_index($cond_field, @fields);

		# Delete matching records
		my (@rec_fields, $updated_field_index);
		for (my $i = 0; $i < scalar(@records); $i++) {
			@rec_fields = split "\t", $records[$i], scalar(@fields);
			if (lc($rec_fields[$field_index]) eq lc($cond_value)) {
				splice(@records, $i--, 1);
			}
		}
	}

	# Save modified table records
	$self->_update_records(@records);
}

# Commit changes to records
sub commit_changes
{
	my ($self, %arg) = @_;

	# Write updated records
	my @records = $self->_get_records();
	foreach my $rec(@records) {
		print DB "$rec\n";
	}
}

# Delete file
sub delete_file
{
	my ($self, %arg) = @_;

	# Error checking
	if (!$arg{'-file_name'}) {
		croak "Incorrect use of lib::FileDB::delete_file() method: missing or invalid -file_name parameter.";
	}

	unlink "$arg{'-file_name'}" || lib::MsgBox->new(text => "Could not delete $arg{'-file_name'}: $!");
}

1;