package lib::Date;
$VERSION = 1.00;
use CGI::Carp qw(fatalsToBrowser);
use strict;

{
# Encapsulated class data

	my @_days_in_month = (31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);

	my @_month_string  = ("Jan", "Feb", "Mar", "Apr", "May", "Jun", 
	                      "Jul", "Aug", "Sep", "Oct", "Nov", "Dec");

	my @_days_string   = ("Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat");

# Class methods, to operate on encapsulated class data

	# Given numeric month value (1-12) and year (optional), 
	# return the number of days in that month
	sub _days_in_month
	{
		my ($self, $month, $year) = @_;

		# Leap year check
		if ($year && $month == 2 && ($year % 4) == 0) {
			return 29;
		} elsif ($month) {
			return $_days_in_month[$month - 1];
		}
	}

	# Given the numeric month value (1-12), return the 
	# string representation of the month
	sub _month_to_string
	{
		my ($self, $month) = @_;

		return $_month_string[$month - 1];
	}

	# Given the numeric day value (1-7), return the 
	# string representation of the day
	sub _day_to_string
	{
		my ($self, $day) = @_;

		return $_days_string[$day - 1];
	}
}

# Constructor
sub new
{
	my ($class, %arg) = @_;
	my $self = bless {}, $class;
}

# Return current date
sub current_date
{
	my ($self, %arg) = @_;

	# Default values
	$arg{'-format'} = "numeric" if !$arg{'-format'};

	my $current_date = time();

	# Return numeric date
	if ($arg{'-format'} eq "numeric") {
		return $current_date;
	} else {
		return $self->to_string(-date => $current_date);
	}
}

# Given numeric date, return corresponding year
sub get_year
{
	my ($self, %arg) = @_;

	# Error checking
	if ($arg{'-date'} eq "") {
		croak "Incorrect use of lib::Date::get_year() method: missing or invalid -date parameter.";
	}

	# Default values
	$arg{'-gmt_offset'} = $main::GM_TIME_OFFSET if !$arg{'-gmt_offset'};

	$arg{'-date'} += ($arg{'-gmt_offset'} * 3600) unless $arg{'-gmtime'};

	my @timery = gmtime($arg{'-date'});

	# Return year
	return ($timery[5] + 1900);
}

# Given numeric date, return corresponding month
sub get_month
{
	my ($self, %arg) = @_;

	# Error checking
	if ($arg{'-date'} eq "") {
		croak "Incorrect use of lib::Date::get_month() method: missing or invalid -date parameter.";
	}

	# Default values
	$arg{'-format'} = 'numeric' if !$arg{'-format'};
	$arg{'-gmt_offset'} = $main::GM_TIME_OFFSET if !$arg{'-gmt_offset'};

	$arg{'-date'} += ($arg{'-gmt_offset'} * 3600) unless $arg{'-gmtime'};

	my @timery = gmtime($arg{'-date'});

	# Return month
	if ($arg{'-format'} eq "string") {
		return $self->_month_to_string($timery[4] + 1);
	} else {
		return ($timery[4] + 1);
	}
}

# Given numeric date, return corresponding day
sub get_day
{
	my ($self, %arg) = @_;

	# Error checking
	if ($arg{'-date'} eq "") {
		croak "Incorrect use of lib::Date::get_day() method: missing or invalid -date parameter.";
	}

	# Default values
	$arg{'-format'} = 'numeric' if !$arg{'-format'};
	$arg{'-gmt_offset'} = $main::GM_TIME_OFFSET if !$arg{'-gmt_offset'};

	$arg{'-date'} += ($arg{'-gmt_offset'} * 3600) unless $arg{'-gmtime'};

	my @timery = gmtime($arg{'-date'});

	# Return day
	if ($arg{'-format'} eq "string") {
		return $self->_day_to_string($timery[6] + 1);
	} else {
		return ($timery[3]);
	}
}

# Given numeric date, return corresponding hour
sub get_hour
{
	my ($self, %arg) = @_;

	# Error checking
	if ($arg{'-date'} eq "") {
		croak "Incorrect use of lib::Date::get_hour() method: missing or invalid -date parameter.";
	}

	# Default values
	$arg{'-gmt_offset'} = $main::GM_TIME_OFFSET if !$arg{'-gmt_offset'};

	$arg{'-date'} += ($arg{'-gmt_offset'} * 3600) unless $arg{'-gmtime'};

	my @timery = gmtime($arg{'-date'});

	# Return hour
	return ($timery[2]);
}

# Given numeric date, return corresponding minute
sub get_minute
{
	my ($self, %arg) = @_;

	# Error checking
	if ($arg{'-date'} eq "") {
		croak "Incorrect use of lib::Date::get_minute() method: missing or invalid -date parameter.";
	}

	# Default values
	$arg{'-gmt_offset'} = $main::GM_TIME_OFFSET if !$arg{'-gmt_offset'};

	$arg{'-date'} += ($arg{'-gmt_offset'} * 3600) unless $arg{'-gmtime'};

	my @timery = gmtime($arg{'-date'});

	# Return minute
	return ($timery[1]);
}

# Given numeric date, return corresponding second
sub get_second
{
	my ($self, %arg) = @_;

	# Error checking
	if ($arg{'-date'} eq "") {
		croak "Incorrect use of lib::Date::get_second() method: missing or invalid -date parameter.";
	}

	# Default values
	$arg{'-gmt_offset'} = $main::GM_TIME_OFFSET if !$arg{'-gmt_offset'};

	$arg{'-date'} += ($arg{'-gmt_offset'} * 3600) unless $arg{'-gmtime'};

	my @timery = gmtime($arg{'-date'});

	# Return second
	return ($timery[0]);
}

# Given numeric date, return corresponding time
sub get_time
{
	my ($self, %arg) = @_;

	# Error checking
	if ($arg{'-date'} eq "") {
		croak "Incorrect use of lib::Date::get_second() method: missing or invalid -date parameter.";
	}

	# Default values
	$arg{'-format'} = $main::TIME_FORMAT if !$arg{'-format'};

	my ($hour, $min, $sec) = ($self->get_hour(-date => $arg{'-date'}),
	                          $self->get_minute(-date => $arg{'-date'}),
	                          $self->get_second(-date => $arg{'-date'}));

	my ($time_string, $ampm);
	if ($arg{'-format'} eq 'ampm') {
		if ($hour > 11 && $hour < 24) {
			$ampm = "PM";
		} else {
			$ampm = "AM";
		}
		if ($hour > 12) {
			$hour -= 12;
		} elsif ($hour == 0) {
			$hour += 12
		}
		$time_string = sprintf("%02d:%02d $ampm", $hour, $min);
	} else {
		$time_string = sprintf("%02d:%02d", $hour, $min);
	}
	$time_string .= " $main::TIME_ZONE" if $main::TIME_ZONE;

	return $time_string;
}

# Given a date, return day, month and year
sub to_string
{
	my ($self, %arg) = @_;

	# Error checking
	if ($arg{'-date'} eq "") {
		croak "Incorrect use of lib::Date::to_string() method: missing or invalid -date parameter.";
	}

	# Default values
	$arg{'-format'} = $main::DATE_FORMAT if !$arg{'-format'};

	my ($year, $month_string, $month, $day) = ($self->get_year(   -date   => $arg{'-date'}),
	                                           $self->get_month(  -date   => $arg{'-date'}, 
	                                                              -format => "string"),
	                                           $self->get_month(  -date   => $arg{'-date'}),
	                                           $self->get_day(    -date   => $arg{'-date'})
	                                          );
	my $date_string = "";
	if ($arg{'-format'} eq 'us_standard') {
		$year = substr($year, 2, 2);
		$date_string = sprintf("%02d/%02d/%02d", $month, $day, $year);
	} elsif ($arg{'-format'} eq 'us_extended') {
		$date_string = "$month_string $day, $year";
	} elsif ($arg{'-format'} eq 'euro_standard') {
		$year = substr($year, 2, 2);
		$date_string = sprintf("%02d/%02d/%02d", $day, $month, $year);
	} elsif ($arg{'-format'} eq 'euro_extended') {
		$date_string =  "$day $month_string, $year";
	}

	if ($arg{'-show_time'}) {
		$date_string .= " ";
		$date_string .= $self->get_time(-date => $arg{'-date'});
	}

	return $date_string;
}

# Given day, month, year and time, return the numeric date representation
sub to_numeric
{
	my ($self, %arg) = @_;

	# Error checking
	if (!$arg{'-year'}) {
		croak "Incorrect use of lib::Date::to_numeric() method: missing or invalid -year parameter.";
	}
	if (!$arg{'-month'}) {
		croak "Incorrect use of lib::Date::to_numeric() method: missing or invalid -month parameter.";
	}
	if (!$arg{'-day'}) {
		croak "Incorrect use of lib::Date::to_numeric() method: missing or invalid -day parameter.";
	}

	# Default values
	$arg{'-hour'} = 0 if $arg{'-hour'} eq "";
	$arg{'-min'}  = 0 if $arg{'-min'}  eq "";
	$arg{'-sec'}  = 0 if $arg{'-sec'}  eq "";
	$arg{'-gmt_offset'} = $main::GM_TIME_OFFSET if !$arg{'-gmt_offset'};

	# Initialize date
	my $date = 0;

	# Add years passed to date (31536000 secs in reg. year, 31622400 secs in leap year)
	for (my $i = 1970; $i < $arg{'-year'}; $i++) {
		if (($i % 4) == 0) {
			$date += 31622400;
		} else {
			$date += 31536000;
		}
	}

	# Add months passed (86400 secs = 1 day)
	for (my $j = 1; $j < $arg{'-month'}; $j++) {
		$date += ($self->_days_in_month($j, $arg{'-year'}) * 86400);
	}

	# Add days passed (86400 secs = 1 day)
	for (my $k = 1; $k < $arg{'-day'}; $k++) {
		$date += 86400;
	}

	# Add hours, minutes and seconds
	$date += (($arg{'-hour'} * 3600) + ($arg{'-min'} * 60) + $arg{'-sec'});

	# GM time adjustment
	$date -= ($arg{'-gmt_offset'} * 3600) unless $arg{'-gmtime'};

	return $date;
}

# Generate a year drop down selection box
sub year_selection_box
{
	my ($self, %arg) = @_;

	# Error checking
	if ($arg{'-start'} !~ /^\d+$/) {
		croak "Incorrect use of lib::Date::year_selection_box() method: missing or invalid -start parameter.";
	}
	if ($arg{'-end'} !~ /^\d+$/) {
		croak "Incorrect use of lib::Date::year_selection_box() method: missing or invalid -end parameter.";
	}

	# Initialize
	my $box_code = qq(<select name="$arg{'-name'}" size="1" class="guiformfield">
                          <option value="Year">Year</option>\n);

	# Add all years
	for (my $i = $arg{'-start'}; $i <= $arg{'-end'}; $i++) {
		if ($arg{'-selected'} == $i) {
			$box_code .= qq(<option value="$i" selected>$i</option>\n);
		} else {
			$box_code .= qq(<option value="$i">$i</option>\n);
		}
	}

	# Finalize
	$box_code .= "</select>\n";

	return $box_code;
}

# Generate a month drop down selection box
sub month_selection_box
{
	my ($self, %arg) = @_;

	# Initialize
	my $box_code = qq(<select name="$arg{'-name'}" size="1" class="guiformfield">
                          <option value="Month">Month</option>\n);

	# Add all months
	my $tmp_month;
	for (my $i = 1; $i <= 12; $i++) {
		$tmp_month = $self->_month_to_string($i);
		if ($arg{'-selected'} == $i || $arg{'-selected'} eq $tmp_month) {
			$box_code .= qq(<option value="$i" selected>$tmp_month</option>\n);
		} else {
			$box_code .= qq(<option value="$i">$tmp_month</option>\n);
		}
	}

	# Finalize
	$box_code .= "</select>\n";

	return $box_code;
}

# Generate a day drop down selection box
sub day_selection_box
{
	my ($self, %arg) = @_;

	# Initialize
	my $box_code = qq(<select name="$arg{'-name'}" size="1" class="guiformfield">
                          <option value="Day">Day</option>\n);

	# Add all days
	for (my $i = 1; $i <= 31; $i++) {
		if ($arg{'-selected'} == $i) {
			$box_code .= qq(<option value="$i" selected>$i</option>\n);
		} else {
			$box_code .= qq(<option value="$i">$i</option>\n);
		}
	}

	# Finalize
	$box_code .= "</select>\n";

	return $box_code;
}

# Generate an hour drop down selection box
sub hour_selection_box
{
	my ($self, %arg) = @_;

	# Initialize
	my $box_code = qq(<select name="$arg{'-name'}" size="1" class="guiformfield">
                          <option value="hour">hour</option>\n);

	# Add all hours
	for (my $i = 0; $i <= 23; $i++) {
		if ($arg{'-selected'} == $i) {
			$box_code .= sprintf("<option value=\"%d\" selected>%02d</option>\n", $i, $i);
		} else {
			$box_code .= sprintf("<option value=\"%d\">%02d</option>\n", $i, $i);
		}
	}

	# Finalize
	$box_code .= "</select>\n";

	return $box_code;
}

# Generate a minute drop down selection box
sub minute_selection_box
{
	my ($self, %arg) = @_;

	# Initialize
	my $box_code = qq(<select name="$arg{'-name'}" size="1" class="guiformfield">
                          <option value="min">min</option>\n);

	# Add all minutes
	for (my $i = 0; $i <= 59; $i++) {
		if ($arg{'-selected'} == $i) {
			$box_code .= sprintf("<option value=\"%d\" selected>%02d</option>\n", $i, $i);
		} else {
			$box_code .= sprintf("<option value=\"%d\">%02d</option>\n", $i, $i);
		}
	}

	# Finalize
	$box_code .= "</select>\n";

	return $box_code;
}

# Generate a minute drop down selection box
sub second_selection_box
{
	my ($self, %arg) = @_;

	# Initialize
	my $box_code = qq(<select name="$arg{'-name'}" size="1" class="guiformfield">
                          <option value="sec">sec</option>\n);

	# Add all seconds
	for (my $i = 0; $i <= 59; $i++) {
		if ($arg{'-selected'} == $i) {
			$box_code .= sprintf("<option value=\"%d\" selected>%02d</option>\n", $i, $i);
		} else {
			$box_code .= sprintf("<option value=\"%d\">%02d</option>\n", $i, $i);
		}
	}

	# Finalize
	$box_code .= "</select>\n";

	return $box_code;
}

# Given the month (1-12) and year (optional), return the number of days in that month
sub days_in_month
{
	my ($self, %arg) = @_;

	# Error checking
	if ($arg{'-month'} !~ /^\d+$/) {
		croak "Incorrect use of lib::Date::days_in_month() method: missing or invalid -month parameter.";
	}

	return $self->_days_in_month($arg{'-month'}, $arg{'-year'});
}

1;