<?php

// +----------------------------------------------------------------------+
// | Opera Hotlist To HTML                                                |
// +----------------------------------------------------------------------+
// | Copyright (c) 2002 Thaiko Kunst <software@thaiko.de>                 |
// +----------------------------------------------------------------------+
// | The purpose of this class is to create an html output out of an      |
// | Opera hotlist file on the fly.                                       |
// |                                                                      |
// | You can just upload your Opera hotlist file and the bookmarks on     |
// | your homepage are automatically up to date. No need for changing     |
// | any other files every time.                                          |
// |                                                                      |
// | Here is a sample:                                                    |
// |     include("OperaHotlistToHtml.php");                                |
// |     $operahtml = new OperaHotlistToHtmlClass();                      |
// |     $htmlcode = $operahtml->ConvertToHtml("opera6.adr");             |
// |     echo "My Bookmarks:<br>\n$htmlcode";                             |
// |                                                                      |
// | With the "templateXXX" functions you can adjust the structure and    |
// | design of the output html code.                                      |
// |                                                                      |
// | If you set "linksafterheader" to FALSE then the sort function is no  |
// | longer in use and the urls and folders are listed in the same order  |
// | as they appear in the hotlist file. Otherwise a simple sort is done  |
// | by listing the urls of a folder prior the subfolders.                |
// |                                                                      |
// | Set the "folderfilter" to filter out folders matching this pattern   |
// | (used by preg_match). The subfolders stay in the list if they do not |
// | the filter.                                                          |
// |                                                                      |
// | Feel free to contact me for suggestions/fixes/...                    |
// +----------------------------------------------------------------------+
// | This library is free software; you can redistribute it and/or        |
// | modify it under the terms of the GNU Lesser General Public           |
// | License as published by the Free Software Foundation; either         |
// | version 2.1 of the License, or (at your option) any later version.   |
// |                                                                      |
// | This library is distributed in the hope that it will be useful,      |
// | but WITHOUT ANY WARRANTY; without even the implied warranty of       |
// | MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    |
// | Lesser General Public License for more details.                      |
// +----------------------------------------------------------------------+
// | Version History:                                                     |
// | v1.0  2002-04-17 .TK.                                                |
// |       First public release                                           |
// | v1.1  2002-04-20 .TK.                                                |
// |       NEW: Filter folders including its subfolders ($filterwithsubs) |
// |       CHG: The comments are now in the PHPDOC format                 |
// |       FIX: Two $tpl_endlist output removed                           |
// +----------------------------------------------------------------------+
// | ToDo:                                                                |
// +----------------------------------------------------------------------+

/** 
 * Create an html output out of an Opera hotlist file on the fly

 * @author Thaiko Kunst <software@thaiko.de>
 * @version 1.1
 * @since 1.0
 * @access public
 * @copyright Thaiko Kunst
 */

class OperaHotlistToHtmlClass {
	//---Public vars---
	/**
	 * Header of the bookmark list. Placeholder: {TITLE}
	 *
	 * @var String
	 * @access public
	 * @see $tpl_header
	 */
	var $tpl_header = "<h1>{TITLE}</h1>\n";

	/**
	 * Footer of the bookmark list. Placeholder: {TITLE}
	 *
	 * @var String
	 * @access public
	 * @see $tpl_footer
	 */
	var $tpl_footer = "";

	/**
	 * Folder Name/Header. Placeholder: {FOLDERNAME}
	 *
	 * @var String
	 * @access public
	 * @see $tpl_entry
	 */
	var $tpl_folder = "  <blockquote><h3><a name=\"{FOLDERNAME}\"></a>{FOLDERNAME}</h3>\n";

	/**
	 * Entry Name. Placeholders: {URL} {BOOKMARKNAME}
	 *
	 * @var String
	 * @access public
	 * @see $tpl_folder
	 */
	var $tpl_entry = "    <a href=\"{URL}\">{BOOKMARKNAME}</a><br>\n";

	/**
	 * Begin of the entry list (between folder title and first entry)
	 *
	 * @var String
	 * @access public
	 * @see $tpl_endlist
	 */
	var $tpl_beginlist = "";

	/**
	 * End of the entry list (after the last entry)
	 *
	 * @var String
	 * @access public
	 * @see $tpl_begilist
	 */
	var $tpl_endlist = "  </blockquote>\n";

	/**
	 * The link in the header to the folder list at the end
	 *
	 * @var String
	 * @access public
	 */
	var $tpl_fldlistlink = "<p><a href=\"#FolderList\">Folder list</a></p>\n<hr>\n";

	/**
	 * Header of the folderlist at the end
	 *
	 * @var String
	 * @access public	
	 * @see $tpl_fldlistfooter
	 */
	var $tpl_fldlistheader = "<hr>\n<h4><a name=\"FolderList\"></a>Folder list:</h4>\n<p style=\"text-align:justify\">\n";

	/**
	 * Footer of the folderlist
	 *
	 * @var String
	 * @access public
	 * @see $tpl_fldlistheader
	 */
	var $tpl_fldlistfooter = "</p>\n<hr>\n";

	/**
	 * An Entry of the footerlist. Placeholder: {FOLDER}
	 *
	 * @var String
	 * @access public
	 * @see $tpl_fldlistheader
	 * @see $tpl_fldlistfooter
	 */
	var $tpl_fldlistentry = "[<a href=\"#{FOLDER}\">{FOLDER}</a>]";

	/**
	 * Output entries directly after their header. FALSE = unsorted.
	 *
	 * @var Boolean
	 * @access public
	 */
	var $linksafterheader = TRUE;

	/**
	 * preg_match-pattern for filtering out folders (the subfolders stay in)
	 *
	 * @var Boolean
	 * @access public
	 */
	var $folderfilter = "";

	/**
	 * If a folder matches $folderfilter all subfolders will be filtered too.
	 *
	 * @var Boolean
	 * @access public
	 */
	var $filterwithsubs = TRUE;

	//---Private vars---
	var $filename = "";
	var $file = 0;
	var $folders = array();
	var $output = "";
	var $fld_personalbar = "Personal bar";
	var $fld_findresults = "Find results";
	var $dofilter = FALSE;


	//-------------------------------------------------------------------------
	//---Public functions------------------------------------------------------
	//-------------------------------------------------------------------------

	/**
	 * Convert the given hotlist file into HTML code.
	 *
	 * Returns the Opera hotlist as HTML code created by using the "templates
	 * (tpl_xxx vars).
	 * Can be collaed more than once for different hotlist files.
	 *
	 * @param string $hotlistfile URL of the Opera hotlist file
	 * @param boolean $withfolderlist Enables/disables the list of folders at the end
	 * @param string $customtitle Title of the output bookmark list
	 * @return string
	 * @access public
	 */
	function ConvertToHtml( $hotlistfile, $withfolderlist = FALSE, $customtitle = "Opera Hotlist" ) {
		// Check file
		if ((trim($hotlistfile) == "") || !file_exists($hotlistfile)) return FALSE;
		$this->filename = $hotlistfile;

		// Init variables
		$this->folders = array();
		$this->output = "";
		$arr = array();
		$dofilter = (strlen($this->folderfilter) > 0);

		// Load bookmark file
		$this->file = fopen($this->filename, "r");
		if ($this->file === FALSE) return FALSE;

		// Check if valid bookmark file
	   	$line = $this->getNextLine();
		if ($line != "Opera Hotlist version 2.0") {
			fclose($this->file);
			return FALSE;
		}

		if (strlen($this->tpl_header) > 0) {
	    	$this->addLine(str_replace('{TITLE}', $customtitle, $this->tpl_header));
	    }
	    if ($withfolderlist) $this->addLine($this->tpl_fldlistlink);
	    $this->addLine($this->tpl_beginlist);

	    $arr = $this->parseFolder(FALSE);

		fclose($this->file);

		$this->outputArray($arr);

		if (strlen($this->tpl_footer) > 0) {
	    	$this->addLine(str_replace('{TITLE}', $customtitle, $this->tpl_footer));
	    }

	    if ($withfolderlist) {
	    	$this->addLine($this->tpl_fldlistheader);

			while (list($key, $folder) = each($this->folders)) {
				$this->addLine(str_replace("{FOLDER}", $folder, $this->tpl_fldlistentry));
			}
			$this->addLine($this->tpl_fldlistfooter);
		}

	    return $this->output;
	}

	/**
	 * Set all template variables for the html output at once.
	 *
	 * Set all template variables (tpl_xxx) which are used for the html 
	 * output at once.
	 * Note that some of the template vars uses placeholders like {TITLE}.
	 * e.g. $tpl_arr = array("tpl_header" -> "<h1>{TITLE}</h1>\n", ...);
	 *
	 * @param array $tpl_arr Array of the template vars
	 * @access public
	 */
	function setTemplateVars($tpl_arr) {
		$this->tpl_header        = $tpl_arr["tpl_header"];
		$this->tpl_footer        = $tpl_arr["tpl_footer"];
		$this->tpl_folder        = $tpl_arr["tpl_folder"];
		$this->tpl_entry         = $tpl_arr["tpl_entry"];
		$this->tpl_beginlist     = $tpl_arr["tpl_beginlist"];
		$this->tpl_endlist       = $tpl_arr["tpl_endlist"];
		$this->tpl_fldlistlink   = $tpl_arr["tpl_fldlistlink"];
		$this->tpl_fldlistheader = $tpl_arr["tpl_fldlistheader"];
		$this->tpl_fldlistfooter = $tpl_arr["tpl_fldlistfooter"];
		$this->tpl_fldlistentry  = $tpl_arr["tpl_fldlistentry"];
	}

	//-------------------------------------------------------------------------
	//---Private functions-----------------------------------------------------
	//-------------------------------------------------------------------------

	/**
	 * Returns the next line of the file
	 *
	 * @return string
	 * @access private
	 */
	function getNextLine() {
		if (feof($this->file)) {
			return "";
		} else {
			return trim(utf8_decode(fgets($this->file, 4096)));
		}
	}

	/**
	 * Adds the line to the output code
	 *
	 * @param string $line
	 * @access private
	 */
	function addLine( $line ) {
		$this->output .= $line;
	}

	/**
	 * Parse the next URL from the hotlist and returns an output line
	 *
	 * @return string
	 * @access private
	 */
	function parseURL () {

		$line = $this->getNextLine();

		$bookmarkname = "";
		$url = "";
		while (!feof($this->file) and ($line != "")) {
			if (!(strpos($line, "=") === FALSE)) {
				list($name, $value) = split("=", $line, 2);
				if ($name == "NAME") $bookmarkname = $value;
				if ($name == "URL") $url = $value;
			}
	    	$line = $this->getNextLine();
		}

		$bookmarkname = htmlentities($bookmarkname);

		return preg_replace(array('({URL})', '({BOOKMARKNAME})'),
		       				array($url, $bookmarkname),
		       				$this->tpl_entry);
	}

	function add_lah(&$arr, $line) {
		if ($this->linksafterheader) {
			array_push($arr, $line);
		} else {
			$this->addLine($line);
		}
	}

	/**
	 * Parse all the folder entries and the subfolders
	 *
	 * @param boolean $withfolder if the folder data shuold be added to the output
	 * @return array
	 * @access private
	 */
	function parseFolder( $withfolder = TRUE ) {

		$arr = array();
		$arrf = array();
		$filtered = FALSE;
		$filtercount = 0;

		if ($withfolder) {
			// Seek to the next folder line
			$foldername = "";
			$line = $this->getNextLine();
			while (!feof($this->file) and ($line != "")) {
				if (!(strpos($line, "=") === FALSE)) {
					list($name, $value) = split("=", $line);
					if ($name == "NAME") $foldername = $value;
				}
				$line = $this->getNextLine();
			} //while
			$foldername = htmlentities($foldername);

			// Check if foldername matches the folderfilter
			if ($dofilter) {
				$filtered = preg_match($this->folderfilter, $foldername) ||
				            preg_match("/^$this->fld_findresults|^$this->fld_personalbar/", $foldername);
			} else {
				$filtered = preg_match("/^$this->fld_findresults|^$this->fld_personalbar/", $foldername);
			}

			if (!$filtered) {
				// Folder header
				array_push($this->folders, $foldername);
				$this->add_lah($arr, str_replace('{FOLDERNAME}', $foldername, $this->tpl_folder));
				$this->add_lah($arr, $this->tpl_beginlist);
			}
		}

		if ($this->linksafterheader && !$filtered) {
			array_push($arr, $arrf);
			$idx = count($arr) - 1;
		}

		// Loop throught the lines
	    while (!feof($this->file)) {
	    	$line = $this->getNextLine();
	    	switch ($line) {
				case "#FOLDER":
					if ($filtered && $this->filterwithsubs) {
						$filtercount++;
					} else {
						array_push($arr, $this->parseFolder());
					}
					break;
				case "#URL":
					if (!$filtered) $this->add_lah($arrf, $this->parseURL());
					break;
				case "-":
					if ($filtered && $this->filterwithsubs && ($filtercount > 0)) {
						$filtercount--;
						break;
					} else {
						break 2;
					}
	    	}
	    }

		// Folder footer
		if ($withfolder && !$filtered) $this->add_lah($arr, $this->tpl_endlist);

		if ($this->linksafterheader && !$filtered) $arr[$idx] = $arrf;
		return $arr;
	}

	/**
	 * Parse the complete hotlist file
	 *
	 * @access private
	 */
	function parse() {
		while (!feof($this->file)) {
	    	$line = $this->getNextLine();
			switch ($line) {
				case "#FOLDER":
					$this->parseFolder();
					break;
				case "#URL":
					$this->parseURL();
					break;
			}
		}
	}

	/**
	 * Write the stored data to the output var
	 *
	 * @param array $outarray the array holding the formatted html lines
	 * @access private
	 */
	function outputArray( $outarray ) {
		while (list($name, $value) = each($outarray)) {
			if (!is_array($value)) {
				$this->addLine($value);
			} else {
				$this->outputArray($value);
			}
		}
	}

}

?>
