#!/usr/local/bin/php -q
<?php
####---------------------------------------------------------------------------
####  hourly_maint.php
####
####  This script is designed to be run at the top of the hour.  It reloads
####  the hourly target tables and loads shared memory with various lookup
####  tables for the delivery engine.
####
####  If called during the 0 hour, it will also call the daily maintenance
####  script.
####
####  If called with the argument "reload", this script assumes it is
####  being called by an admin.  It will not process log files, and it
####  _will_ call the daily maintenance script.
####
####  Since the changes to shared memory must be done safely, there can be
####  a delay introduced in the delivery engine while this script holds on
####  to a semaphore and loads shared memory.
####---------------------------------------------------------------------------

set_time_limit(0);

require_once("lib-envvar.inc");
require("lib-oasis.inc");
require("lib-db.inc");
require("lib-maint.inc");
require("lib-dengine.inc");
require("lib-report.inc");
require("lib-logs.inc");
require_once("lib-i18n.inc");

$runtimeopts = array(
                'stop'        => array('record_last' =>1,   'rename_logs'  =>1,
                                       'process_logs'=>1,   'simulate'     =>0,
                                       'clear_shm'   =>1,   'reload_hourly'=>0,
                                       'call_daily'  =>0,   'force_daily'  =>0),
                'resume'      => array('record_last' =>0,   'rename_logs'  =>0,
                                       'process_logs'=>0,   'simulate'     =>0,
                                       'clear_shm'   =>0,   'reload_hourly'=>1,
                                       'call_daily'  =>0,   'force_daily'  =>0),
                'start'       => array('record_last' =>0,   'rename_logs'  =>0,
                                       'process_logs'=>0,   'simulate'     =>0,
                                       'clear_shm'   =>0,   'reload_hourly'=>1,
                                       'call_daily'  =>1,   'force_daily'  =>1),
                'reload'      => array('record_last' =>1,   'rename_logs'  =>1,
                                       'process_logs'=>1,   'simulate'     =>0,
                                       'clear_shm'   =>1,   'reload_hourly'=>1,
                                       'call_daily'  =>0,   'force_daily'  =>0),
                'force_daily' => array('record_last' =>1,   'rename_logs'  =>0,
                                       'process_logs'=>0,   'simulate'     =>0,
                                       'clear_shm'   =>1,   'reload_hourly'=>1,
                                       'call_daily'  =>1,   'force_daily'  =>1),
                'default'     => array('record_last' =>1,   'rename_logs'  =>1,
                                       'process_logs'=>1,   'simulate'     =>1,
                                       'clear_shm'   =>1,   'reload_hourly'=>1,
                                       'call_daily'  =>1,   'force_daily'  =>0)
                );

if(!($rtopt = $argv[1])) $rtopt = 'default';
$action = $runtimeopts[$rtopt];

admin_log("runtime options: $rtopt");

$now = time();
$hour = date('G', $now);
read_traffic_profile();

$log_dir = get_prefs('LogDir');

if($hour == 0 && $action[rename_logs])
{
  #### don't subtract a full date, since at DST, you'll get the day
  #### _before_ yesterday!  Just subtract half a day.
  $log_date = date('Y/m-d', $now - 43200);

  if(!@rename("$log_dir/oasis_admin.log", "$log_dir/$log_date-admin.log"))
    warn("Could not rename $log_dir/oasis_admin.log to $log_dir/$log_date-admin.log: $php_errormsg\n");

  if(!@rename("$log_dir/oasis_sim.log", "$log_dir/$log_date-sim.log"))
    warn("Could not rename $log_dir/oasis_sim.log to $log_dir/$log_date-sim.log: $php_errormsg\n");
}

#### throw the system into overflow mode and reload the primary delivery tables
set_overflow_flag (1);

$sem_start_time = time();

$OG_active_creatives = array();
if($action[record_last]) record_last_hour($hour);

#### call the daily script if this is the zero hour
if(($hour == 0 && $action[call_daily]) || $action[force_daily])
{
  admin_log("running daily maintenance code...");

  if(!$action[force_daily]) check_for_underdelivery();

  compute_daily_targets('', '');
}
load_creative_content('');

if($action[reload_hourly])
{
  $wiped_clean = array();

  admin_log("building assignment table...");
  build_assignment_table(0, $hour);

  admin_log("computing next hour's targets...");
  compute_hourly_targets($hour, '');
}

if($action[clear_shm]) clear_shared_memory();

#### now populate the shared memory, if we've got something
if($action[reload_hourly]) load_shared_memory();

sem_release($OG_sem_id);

$sem_elapsed_time = time() - $sem_start_time;
admin_log("overflow mode lasted $sem_elapsed_time seconds");

#### move the logs (which contain primary deliveries and overflows)
if($action[rename_logs])
{
  #### it's possible that the log file does not exist, so check before
  #### attempting to move the file.
  if(file_exists("$log_dir/oasis.log"))
  {
    $year = date('Y', $now - 3600);
    $mon  = date('m', $now - 3600);
    $day  = date('d', $now - 3600);
    $hour = date('H', $now - 3600);

    @mkdir("$log_dir/$year", 0755);

    #### pause all delivery while we move the logs
    if(!(($sem_id = sem_get(0x4f410000, 1)) && sem_acquire($sem_id))) {
        admin_log("FATAL ERROR: could not acquire semaphore 0x4f410000");
        exit;
    }
    admin_log("moving log file to $log_dir/$year/$mon-$day-$hour.log...");
    if(!@rename("$log_dir/oasis.log", "$log_dir/$year/$mon-$day-$hour.log"))
    {
      warn("Could not rename $log_dir/oasis.log to $log_dir/$year/$mon-$day-$hour.log: $php_errormsg\n");
    }
    sem_release ($sem_id);
  }
}

#### now reload the overflow shared memory
set_overflow_flag (0);
load_creative_content('');
if($action[record_last]) record_last_hour($hour);
if($action[clear_shm]) clear_shared_memory();
if($action[reload_hourly]) load_shared_memory();

sem_release($OG_sem_id);



#### do things that don't require semaphore
if($action[process_logs])
{
  if(file_exists("$log_dir/$year/$mon-$day-$hour.log"))
  {
    process_log("$log_dir/$year/$mon-$day-$hour.log",
              "$log_dir/$year/$mon-$day.log");
  }

  if($hour == 0)
  {
    housekeeping();
    if(date('w', $now) == 0) mail_reports();
  }
}

if($hour == 0 && $action[simulate])
{
  admin_log("checking inventory...");
  `nice -10 ./check_inventory.php`;
  admin_log("done.");
}

update_campaigns();
mark_complete();

####---------------------------------------------------------------------------
####---------------------------------------------------------------------------
function clear_shared_memory()
{
  global $OG_shm_hour_ass;

  #### we need to do this periodically to avoid filling shared memory;
  #### since segment OAS 0 is keyed by SectionID (OAS0), over time
  #### it will fill up with unused sections' info.
  
  #### Note that this could be dangerous -- when we were clearing segment
  #### OAS1, we would sometimes get this error:
  ####       shmget() failed for key 0x4f415301: File exists
  #### because of this, we are no longer clearing OAS1; instead, after
  #### recording the previous hour's info for each creative, we
  #### call shm_remove_var().  After looping through all creatives running
  #### during the last hour, we have in effect cleared OAS1.

  admin_log("clearing shared memory...");

  #### if we're running for the first time, there won't be anything in here;
  #### thus we don't call clear_shared_memory() upon startup

  #### php version 4.0.7 changed the shm_remove() semantics
  if(get_php_version() >= 40007)
  {
    if(!@shm_remove(shm_attach($OG_shm_hour_ass,
                        get_prefs('ShmSizeHourlyAssignments'))))
      abort("Error clearing hourly assignment shm ($OG_shm_hour_ass) - $php_errormsg");
  }
  else
  {
    if(!@shm_remove($OG_shm_hour_ass))
      abort("Error clearing hourly assignment shm ($OG_shm_hour_ass) - $php_errormsg");
  }
}



####---------------------------------------------------------------------------
####---------------------------------------------------------------------------
function record_last_hour($hour)
{
  global $OG_active_creatives;
  global $OG_shm_delivered;
  global $OG_overflow, $OG_shm_hour_tar;

  admin_log("recording last hour's deliveries ($OG_overflow)...");

  $lasthour = ($hour == 0) ? 23 : $hour - 1;

  #### look in the hourly targets shared memory segment to determine how many of
  #### each creative was delivered (subtract target from remaining)
  if(!(@$smh = shm_attach($OG_shm_hour_tar, get_prefs('ShmSizeHourlyTargets'))))
    abort("Error connecting to shared memory segment OAS1 ($php_errormsg)\n");

  #### go through all daily targets; look at what has been delivered
  if($result = mysql_query("select CreativeID from RunningCreatives"))
  {
    #### build an array of the number of impressions delivered, hashed on
    #### the creative ids
    $shm_delivered = array();
    while(list($cr_id) = mysql_fetch_row($result))
    {
      if (@$target_info = shm_get_var($smh, $cr_id)) {
        $imps = $target_info[0] - $target_info[2];
        $clicks = $target_info[3];
        $shm_delivered[$cr_id] = array($imps, $clicks);
      
        #### OG_shm_delivered holds the primary plus the overflow, which will
        #### be used when comparing against the logs
        if ($OG_shm_delivered[$cr_id])
            list ($totimps, $totclicks) = $OG_shm_delivered[$cr_id];
        else list ($totimps, $totclicks) = array (0, 0);
        $OG_shm_delivered[$cr_id] = array ($totimps + $imps, $totclicks + $clicks);
        $OG_active_creatives[$cr_id] = 1;
      }
    }

    #### go through the hash, and update the database
    while(list($cr_id, $v) = each ($shm_delivered))
    {
      list($i, $c) = $v;
      if(!$i) $i = 0;
      if(!$c) $c = 0;

      #print "creative $cr_id: $i impressions, $c clicks\n";

      if(!mysql_query("update DailyTargets set Remaining=Remaining-$i where CreativeID=$cr_id"))
        warn("Error updating DailyTargets for creative $cr_id (" . mysql_error() . ")");

      #### update Creatives table
      $sql = "update Creatives set ImpressionsDelivered=ImpressionsDelivered + $i, ClicksDelivered=ClicksDelivered + $c where CreativeID=$cr_id";
      if(!mysql_query($sql))
        abort("Error updating Creatives for creative $cr_id: "
                . mysql_error() . "($sql)\n");

      #### since we've recorded the information, clear the shared memory
      if(!@shm_remove_var($smh, $cr_id))
        warn("Error clearing creative $cr_id's target info ($php_errormsg)");
    }
  }
  else
    abort("Error querying RunningCreatives: " . mysql_error() . "\n");

  if(!@shm_detach($smh))
    abort("Error detaching from hourly targets shm ($OG_shm_hour_tar) - $php_errormsg");

  #### we only want to clear the RunningCreatives table once, while we're
  #### processing primary deliveries
  if ($OG_overflow) mysql_query("delete from RunningCreatives");

  admin_log("hourly deliveries recorded");
}

?>
