package net.sourceforge.idrs.utils.pool;
/*
 *  PoolMan Java Object Pooling and Caching Library
 *  Copyright (C) 2000 The Code Studio
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *  
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *  
 *  The full license is located at the root of this distribution
 *  in the LICENSE file.
 */


import java.util.*;
import java.io.*;


/**
 * ObjectPool provides a mechanism for pooling objects and resources.
 * It must be subclassed to acocmplish specific pools; GenericPool 
 * and JDBCPool are examples of such subclasses.
2 *<br>
 *@see GenericPool
 *@see JDBCPool
 */
public abstract class ObjectPool implements Serializable {
    //Added Marc Boorshtein - 8/31/00
    protected PrintStream out;          //Used for logging pool activity
    protected int maxTrys;
    protected long daysOpen;            //How many days a pooled object will stay open before being rebuilt
    protected long sleepTime;
    protected static final String DEFAULT_PATH = Long.toString(System.currentTimeMillis()) + ".log";  //default path for pooler log
    protected static final int DEFAULT_TRYS = 10;  //default number of tries to extract an object out of the pool before null is returned
    protected static final long DEFAULT_DAYS_OPEN = 30;  //default number of days an object is open before it is rebuilt
    protected static final int DEFAULT_MIN_POOLSIZE = 1;
    protected String logPath; //used for logging the pool
    protected int minn;  //min in pool                         
    //protected Hashtable pool;
    protected PooledObject[] pool;
    //end add

    protected int limit;		// total number of allowed objects
    protected int count;		// current number of created objects

    /* Removed Marc Boorshtein - 8/31/00
    protected long expirationTime;	// time objects have to live
    protected long objectTimeout;       // time SmartObjects have before being returned
    */

   // protected Hashtable locked;		// objects checked out
   // protected Hashtable unlocked;	// objects available for checkout

    /*Removed Marc Boorshtein - 8/31/00
    protected Thread skimmer;		// thread to clean up dead objects
    protected Thread lifeguard;		// thread to protect objects from hazard\
    */

    protected boolean debug = false; 	// whether or not to print diagnostic data
    
    // some default values
    public static final long DEFAULT_EXPIRATION = 600000;	// 10 minutes
    public static final long DEFAULT_SLEEPTIME = 300000;	// 5 minutes
    public static final int MAX_POOLSIZE = Integer.MAX_VALUE;	// System's max
    public static final long DEFAULT_TIMEOUT = 30000;		// 30 seconds
    
    /**
     * Creates an ObjectPool with the default params.
     */
    public ObjectPool() throws Exception {
    	this(DEFAULT_MIN_POOLSIZE,MAX_POOLSIZE, DEFAULT_DAYS_OPEN, DEFAULT_SLEEPTIME, DEFAULT_PATH,DEFAULT_TRYS);
    }
    
    /**
     * Create an ObjectPool with no object count limit.
     */
    public ObjectPool(long daysOpen, long sleepTime) throws Exception {
	this(DEFAULT_MIN_POOLSIZE, MAX_POOLSIZE, daysOpen, sleepTime, DEFAULT_PATH, DEFAULT_TRYS);
    }

    /** 
     * Create an instance of ObjectPool
     * @param expirationTime is the time in milliseconds that each object has to live.
     * @param objectLimit is the total number of objects that are allowed in the pool at any time. If objectLimit is equal to or less than 0, it is assumed that the limit is Integer.MAX_VALUE
     * @param sleepTime is the time in miliseconds that the pool skimming thread sleeps between 'reap' cycles
     */
    public ObjectPool(int objectMin, int objectLimit, long daysOpen, long sleepTime, String path,int trys) throws Exception {
	/*remove Marc Boorshtein - 8/31/00
	this.objectTimeout = timeout;
	this.expirationTime = expirationTime;
	*/
	
	this.limit = objectLimit;

	//added Marc Boorshtein - 8/31/00
    	this.out = new PrintStream(new FileOutputStream(path,false));
    	this.maxTrys = trys;
    	this.daysOpen = daysOpen;
     this.logPath = path;
     this.sleepTime = sleepTime;
     this.pool = new PooledObject[objectLimit];
     //end add
	
	this.count = 0;
	
	if(this.limit <= 0)
	    this.limit=Integer.MAX_VALUE;
	//added Marc Boorshtein - 8/31/00
	
	this.minn = objectMin;
	
	//end add
	
	/*Removed Marc Boorshtein - 8/31/00
	// skimmer removes dead or useless objects
	this.skimmer = new Thread(new PoolSkimmerThread(sleepTime, this));
	skimmer.start();
	
	// lifeguard protects SmartObjects
	this.lifeguard = new Thread(new LifeGuardThread(objectTimeout, this));
	lifeguard.start();
	*/
	
    }

    /* Removed Marc Boorshtein - 8/31/00
    public long getTimeout() { return this.objectTimeout; }
    */
    
    public boolean isDebugging() {
    	return this.debug;
    }
    
    public void setDebugging(boolean b) {
    	this.debug = b;
    }

    //added Marc Boorshtein - 8/31/00
    abstract protected void logHeader() throws Exception;
    //end add

    abstract protected PooledObject create() throws Exception;
    abstract protected boolean validate(PooledObject obj );
    abstract protected void expire( PooledObject obj );
    /*
    /* @return int The total number of objects in the pool, both locked and unlocked. 
    public int numTotalObjects() {
	return (numUnlockedObjects() + numLockedObjects());
    }

    /* @return int The number of objects currently available. 
    public int numUnlockedObjects() {
	return unlocked.size();
    }

    /** @return int The number of objects currently checked out. 
    public int numLockedObjects() {
	return locked.size();
    }

    /* Removed Marc Boorshtein - 8/31/00
    public long getExpirationTime() {
	return this.expirationTime;
    }

    public void setExpirationTime(long l) {
	this.expirationTime = l;
    }
    */

    /** 
     * Checkout cycles through the available objects and returns the first valid
     * object it finds.  if there are no valid objects, checkOut will create
     * one, unless the client specified amount of objects have already been
     * created, and then it will block until it finds an appropriate object
     * @returns Object the object this pool holds
     */
    public Object checkOut(int num) throws Exception {
     
	     boolean foundOne = false;
	  	long now = System.currentTimeMillis();
	  	String key;
	  	PooledObject obj = null;
	  	int i;
	  	//System.out.println("Count : " + count);
	  	//System.out.println("Num : " + num);
	  	if (num <= this.maxTrys) {
	  	  if (count > 0)   {
	  	      
	  	      
	         synchronized (pool) {	
	  	      for (i=0;i<count;i++) {
	  	       //System.out.println("HERE");
	  		  //Object o = pool[i];
	  		  //System.out.println("Object at " + i + " : " + o);
	  		  key = pool[i].getValue().toString();
	  		  //System.out.println("Key + " + key);
	  		  //obj = pool[i]; 
	  		  //System.out.println("Object + " + obj.toString());
	  		  if (pool[i].isFree()) {
	  		    if(validate(pool[i]))  {
	  		        if (shouldReset(i)) {
	  		        	reset(i);
	  		        }
	  		        pool[i].isFree(false);
	  		        //notifyAll();
	  		        //foundOne = true;
	  		        //break;
	  		        return (pool[i].getValue());
	  		    }
	  		    else {
	  		        // object failed validation
	  		        synchronized (this) {
	  		          
     	  		     java.util.Date dnow = new java.util.Date(System.currentTimeMillis());
	                      out.println(dnow.toString() + "---------> " + dnow.toString() + " Object " + obj.getValue().toString() + " has failed validation");
	  		          
	  		          expire(pool[i]);
	  		          pool[i] = create();
	  		          pool[i].isFree(false);
	  		          //pool[i] = obj;
	  		          foundOne = true;
	  		          return (pool[i].getValue());
	                  }		      
   	  		        
	  		    }
	  		  }
	  		 
	  	      }
	  	    }
	  	    
	  	  }
	  	
	  	  // no objects available, create a new one
	  	  if(count < limit) {
	  	      // there is still room for objects
	  	      synchronized (this) {
	  	        obj = create();
	  	        obj.isFree(false);
	  	        count++;
	  	        pool[count] = obj;
	  	        
	  	      /*removed Marc Boorshtein - 8/31/00
	  	      if (o instanceof com.codestudio.util.SmartObject) {
	  		  com.codestudio.util.SmartObject so = (com.codestudio.util.SmartObject) o;
	  		  so.setState(SmartObject.LOCKED);
	  	      }
	  	      */
	  	        
	  	      }
	  	      //notifyAll();
	  	      return obj.getValue();
	  	  }
	  	  else {
	  	      //there is no room for objects
	  	      try {
	  		 Thread.sleep(sleepTime);
	  		   
	  	      } catch(InterruptedException e) {
	  		  System.out.println("ERROR: Failed while waiting for an object:");
	  		  System.out.println(e);
	  	      }
	  	      //changed Marc Boorshtein - 8/31/00
	  	      out.println("---------> All Connections Exhausted, waiting in loop " + num);
     	  	   return checkOut(num + 1); 
	  	  }
	  	}
	  	else {
	  	  System.out.println("Returning Here");
	  	  return null;
	  	}
	 
	 
  }
    
  protected int getConnID(Object obj)  {
    int i;
    boolean found = false;
    for (i=0;i<limit;i++) {
      if (pool[i].equalsObj(obj)) {
        found = true;
        break;
      }
    }
    
    if (found) 
      return i;
    else
      return -1;
    
  }
    /** Checks an object back into the pool. */
    protected  void checkIn(Object obj) throws Exception {
     
        
        PooledObject pobj;
	    if (obj != null) {
	      int pos = getConnID(obj);
	      if (pos == -1)
	        throw new Exception("Trying to return an object that was never part of the pool");
	      pobj = pool[pos];
	      pobj.isFree(true);
	    }
	    else {
	      //for some reason the connection has been lost, we will re-create it
	      /* will come back
	      try {
             pobj = create();
             pobj.isFree(true);
             count++;
             out.println("------> Connection lost, rebuilding");
             System.out.println("Connection lost");
             pool.put(pobj.getValue().toString(),pobj);
           }
           catch (Exception e) {
             out.println("There was an error in creating a connection : " + e);	
           }
           */
           System.out.println("Lost Connection");
	    }
	    
    
     
   }
   
   public void checkIn(String DBName) throws Exception {
     int pos = getConnID(DBName);
     PooledObject obj;
     if (pos == -1) {
       throw new Exception("Connection Lost");
     }
     else {
       if (pool[pos].getValue() != null) {
         pool[pos].isFree(true);
       }
       else {
         out.println("-------> Connection " + pos + " lost, rebuilding");
         synchronized (pool) {
         	 int tmpCount = count;
         	 count = pos;
           pool[pos] = create();
           count = tmpCount;
         }
       }
     }  	
   }
  
  protected boolean shouldReset(int index) { 
    long now = System.currentTimeMillis();
    long built = pool[index].getBorn();
    long maxDaysOpen = this.daysOpen * 1000 * 60 * 60 * 24;
    long beenOpen = now - built;
    return (beenOpen >= maxDaysOpen);
  }
  
  protected abstract void reset(int index);
}

