package net.sourceforge.idrs.utils;

import java.sql.*;
import java.util.*;
import java.text.*;
import net.sourceforge.idrs.script.IDRSScript;
import java.io.*;
/**
 *Class DB
 *DB.java
 *Copyright (C) 2000 Marc Boorshtein under the GNU General Public License offered
 * without warenty
 *
 *Purpose: This class is used as a wrapper for JDBC to insure cross database
 *compatability.  This class allows for the connection to and retrieval from DB's without
 *the need for dirrect use of JDBC
 */

public class DB implements Serializable{
  //public fields
  public  static final int SERVER = 1;
  public final static int CLIENT = 2;

 public static final int INPUT = 1;
  public static final int OUTPUT = 2;
  
  private int cursurLocation, beginWith = 0;
  private int cursur;
  private Hashtable fields;
  private Vector varTypes, vars,rows;
  private String user, driver, name, pass, ID;
  private boolean closeConn;
  ResultSet rs;
  Connection con;
  private boolean firstRead;
  private boolean isPaged;
  private int numRecs;
  private boolean hasNext;
  private int inputResult;
  private boolean wasCached;
  private int direction;
  IDRSScript idrs;
  //***************************************************************************
  //Following methods are used for data-caching                               *
  //***************************************************************************

  /**
   *Sets wether or not a DB was retrieved from a cache
   */
  
  public void setDirection(int dir) {
    this.direction = dir;  
  }
  
  public int getDirection() {
    return this.direction;
  }
  
  public void wasCached(boolean yup) throws Exception {
    wasCached = yup;
    this.cursur = DB.CLIENT;
  }


  /**
   *Returns wether or not the db was retrieved from a cache
   */
  public boolean wasCached() throws Exception {

    return wasCached;
  }


  /**
   * Returns the number of the first record to be retrieved from dataset,
   * used only with client side cursur
   */
  public int getFirstRec() throws Exception {
    return this.beginWith;
  }

  /**
   * Returns the number  of records should be retrieved in current report
   */
  public int getNumRecs() throws Exception {
    return numRecs;
  }


  /**
   * Sets the number of records to be retrieved in a report
   */
  public void setNumRecs(int recs) throws Exception {
    this.numRecs = recs;
  }

  /**
   * Returns wether or not current DB is paged
   */
  public boolean getIsPaged() throws Exception {
    return isPaged;
  }

  /**
   * Sets wether or not the db is paged
   */
  public void setIsPaged(boolean page) throws Exception {
    isPaged = page;
    this.cursur = DB.CLIENT;

  }

  /**
   * Returns the current cursor location of the db
   */
  public int getCurrLocation() throws Exception {
    return this.cursurLocation;
  }



  /*
   *Pre: java.util
   *Post: creates a new DB that is ready for use.  NOTE: db is not ready to connect yet
  */

  public DB(String ID) {
    super();
    varTypes = new Vector();
    vars = new Vector();
    this.ID = ID;
    user = "";
    driver = "";
    name = "";
    pass = "";
    firstRead = false;
    wasCached = false;
    direction = DB.OUTPUT;

  }



  
  public String getID() throws Exception {
    return this.ID;
  }

  /*Added for datapaging*/
  public void setCursurType(int type) throws Exception {
    this.cursur = type;
  }
  //***************************************************************************************/
  //*These methods deal with the initialization of a JDBC connection                      */
  //***************************************************************************************/
  public void setIDRS(IDRSScript idrs) {
    this.idrs = idrs;
  }
  
  public boolean shouldPool() {
    return (! closeConn);
  }
  /**
   * Closes the connection and recordset
   */
  public void close() throws Exception {
    if (closeConn)
      con.close();
    if (this.cursur == DB.SERVER)
      rs.close();

  }
  /**
   *Pre: driver must be in the CLASSPATH
   *Post: dirver is loaded by the VM and is ready for use
  */

  public void setDBDriver(String driver) throws Exception {
    this.driver = driver;
    Class.forName(driver);
  }


  /**
   *returns the name of the driver being used
  */
  public String getDBDriver() throws Exception {
    return this.driver;
  }

  /**
   *Pre: setDBDriver(), setUser(), setPassword() must all be executed
   *Post: object is connected to the specified database
  */
  public void setDBName(String name) throws Exception {

    this.name = name;
    if (user.equalsIgnoreCase(""))
      con = DriverManager.getConnection(name);
    else
      con = DriverManager.getConnection(name,user,pass);
    closeConn = true;
  }

  public void setDBName(String name, boolean connect) throws Exception {

    this.name = name;
    if (connect) {
      con = DriverManager.getConnection(name,user,pass);
      closeConn = true;
    }
  }


  /**
   *Returns the name of the DBName
  */
  public String getDBName() throws Exception {
    return this.name;
  }

  /**
   *Pre:none
   *Post: sets user name, use "" for no user name
  */
  public void setUserName(String user) throws Exception {
    this.user = user;
  }

  /**
   *returns the user name
  */
  public String getUserName() throws Exception {
    return this.user;
  }

  /**
   *Pre: none
   *post: sets passwords, use "" for no password
   */
  public void setPassword(String pass) throws Exception {
    this.pass = pass;
  }

  /**
   *returns the password
  */
  public String getPassword() throws Exception {
    return this.pass;
  }

  /**
   *Is used to set a pre-made connection to a DB
  */
  public void setConnection(Connection con) throws Exception {
    this.con = con;
    closeConn = false;
  }

  /**
   *Is used to retrieve a connection to DB
  */
  public Connection getConnection() throws Exception {
    return this.con;
  }
  //****************************************************************************************/
  //*end JDBC init methods                                                                 */
  //****************************************************************************************/

  //****************************************************************************************/
  //*These methods deal with the loading of a resultset                                    */
  //****************************************************************************************/

  /**
   *Pre:  setDBName() has been executed successfully
   *      SQL is a valid SQL statement OR SQL is the name of a stored procedure
   *      all parameters must be shown in SQL using the ? character
   *      the vars and varTypes vectors must be filled for each parameter in SQL
   *      the vars and varTypes must contain information acording to the order in which
   *      the parametes apear in SQL
   *      if there are no parameters, then vars and varTypes must be empty
   *Post: retrieves information based on SQL
   */
  public void procMethod(ObjectStore obj,String method,Vector vals) throws Exception {
    Vector finalVals = new Vector();
    Enumeration e = vals.elements();
    Class[] clss = obj.getMethodTypes(method);
    Class cls;
    String val;
    Object objVal = new Object();
    int i = 0;
    Object val1;
    
    while (e.hasMoreElements()) {
      cls = clss[i];
      objVal = e.nextElement();
      val = objVal.toString();
      
      
      

        val1 = ObjectStore.getValue(cls,val,this.con,this.idrs);
      cls = val1.getClass();
      finalVals.add(val1);

      i++;
    }
    if (direction == DB.OUTPUT) {
      rs = (ResultSet) obj.execMethod(method,finalVals.toArray());

      if (! (rs == null)) {
        this.readTable();
      }
      else throw new Exception("Method Doesn't Return ResultSet");
    }
    else { //direction is input
      Integer tmp = (Integer) obj.execMethod(method,finalVals.toArray());
      this.inputResult = tmp.intValue();
    }
  }
  public void ProcSQL(String SQL, boolean isProc) throws Exception {
    int i;
    if (isProc) {
      SQL = "{call " + SQL + "}";
      CallableStatement cs = con.prepareCall(SQL);
      if (vars != null) {
        for (i = 1;i <= vars.size();i++) {
          this.setValue(cs, (String) varTypes.elementAt(i - 1),i);
        }
      }
      if (this.direction == DB.OUTPUT) {
        rs = cs.executeQuery();
      }
      else {
        this.inputResult = cs.executeUpdate();	
      }
    }
    else {
      PreparedStatement ps = con.prepareStatement(SQL);
      if (vars != null) {
        for (i = 1;i <= vars.size();i++) {
          this.setValue(ps,(String) varTypes.elementAt(i - 1),i);
        }
      }
      
      
      if (this.direction == DB.OUTPUT) {
        rs = ps.executeQuery();
      }
      else {
        this.inputResult = ps.executeUpdate();	
      }
    }
    if ((this.direction == DB.OUTPUT) && ! (rs == null)) {
      this.readTable();
    }

  }

  /**
   *Pre: (index - 1) < vars.size()
   *post: writes the parameter at (index - 1) to ps in the Database's own format
   */
  private void setValue(PreparedStatement ps, String varType,int index) throws Exception {

    varType = varType.trim();
    if (varType.equalsIgnoreCase("string")) {
      ps.setString(index,(String) vars.elementAt(index - 1));
    }
    else if (varType.equalsIgnoreCase("int")) {
      ps.setInt(index,Integer.parseInt((String) vars.elementAt(index - 1)));
    }
    else if (varType.equalsIgnoreCase("float")) {
      ps.setFloat(index, Float.parseFloat((String) vars.elementAt(index - 1)));
    }
    else if (varType.equalsIgnoreCase("Date")) {
      java.sql.Date dt = new java.sql.Date(java.sql.Date.parse((String) vars.elementAt(index - 1)));
      ps.setDate(index,dt);
    }
    else if (varType.equalsIgnoreCase("time")) {
      java.sql.Time tm = new java.sql.Time(java.sql.Time.parse((String) vars.elementAt(index - 1)));
      ps.setTime(index,tm);
    }

  }





  //**************************************************************************************/
  //*these methods deal with the manipulation of the variables used in an SQL statement  */
  //*or stored procedure                                                                 */
  //**************************************************************************************/

  /**
   *pre:none
   *post: used to add a var to a query
   */
  public void addVar(String val, String type) throws Exception {
    vars.addElement(val);
    varTypes.addElement(type);
  }

  /**
   *pre: index < vars.size();
   *post: removes variable index from vars and varTypes vectors
   */
  public void removeVar(int index) throws Exception {
    vars.removeElementAt(index);
    varTypes.removeElementAt(index);
  }

  /**
   *pre: index < vars.size()
   *post: sets a variable's value and type at index
   */
  public void setVar(String val, String type,int index) throws Exception {
    vars.setElementAt(val,index);
    varTypes.setElementAt(type,index);
  }

  /**
   *Clears all variables
   */
  public void clearVars() throws Exception {
    vars.removeAllElements();
    varTypes.removeAllElements();
  }
  //**************************************************************************************/
  //*End variable manuipulation methods                                                  */
  //*************************************************************************************/

  /**
   * pre: rs is populated
   * Post: populates a vector of hashtables
   */
  private void readTable() throws Exception {

    if (this.cursur == DB.CLIENT) {
      rows = new Vector();
      while (rs.next()) {
        this.fields  = new Hashtable();
        readFields();
        rows.addElement(this.fields);
      }
      rs.close();
      this.cursurLocation = this.beginWith;
    }
    else {//cursor is server by default
      rs.next();
      this.readFields();
      firstRead = false;
    }

  }
  /**
   *pre: rs is popultated
   *post: fields hashtable is populated with the value and name of each field
  */
  private void readFields() throws Exception {
    if (rs != null) {
      int i;
      fields = new Hashtable();
      String colName, colVal = "";
      ResultSetMetaData md = rs.getMetaData();
      int type;
      String sType;
      int cols = md.getColumnCount();
      java.sql.Timestamp td;
      for (i = 1;i <= cols;i++) {
        colName = md.getColumnName(i);
        type = md.getColumnType(i);
        sType = md.getColumnTypeName(i);
        if (sType.equalsIgnoreCase("TIME")) {
          try {
            java.sql.Time tmVal;
            tmVal = rs.getTime(colName);
            colVal = tmVal.toString();
          }
          catch (Exception e) {
            colVal = "";
          }
        }
        else if (sType.equalsIgnoreCase("DATE")) {
          try {
            java.sql.Date dtVal;
            dtVal = rs.getDate(colName);
            colVal = dtVal.toString();
          }
          catch (Exception e) {
            colVal = "";
          }
        }
        if (((sType.indexOf("TIME") != -1) && (sType.indexOf("DATE") != -1)) || ((sType.indexOf("time") != -1) && (sType.indexOf("TIME") != -1))) {
          try {
            td = rs.getTimestamp(colName);
            colVal = td.toString();
          }
          catch (Exception e) {
            colVal = "";
          }
        }
        else
          try {
            colVal = rs.getString(colName);

          }
          catch (Exception e) {
            colVal = "";

          }

        if (rs.wasNull()) {
          colVal = "";

        }

        fields.put(colName,colVal);

      }
    }

  }
  //****************************************************************************************/
  //*End of resultset loading                                                              */
  //****************************************************************************************/

  //****************************************************************************************/
  //*Methods used for retrieval of data from resultset and moving through a resultset      */
  //****************************************************************************************/
  public int getUpdateResult() throws Exception {
    if (this.direction != DB.INPUT)
      throw new Exception("getUpdateResult() can only be called when direction is INPUT");
    return this.inputResult;
  }
  
  /**

   *pre: fields is poulated
   *     fieldname exists in fields
   *	  if no format, then it should be set to null
   *	  format must follow java.text.MessageFormat() specs
   *post: returns the data in fieldname with applied format
   */
  public String getFieldData(String fieldname,String format) throws Exception {
    if (this.direction != DB.OUTPUT)
      throw new Exception("getFieldData() can only be called when direction is OUTPUT");
    
    if (rs == null) {
      return "";
    }
    else {

      String data = "";
      data = (String)  fields.get(fieldname);
      Object mess [] = {new String(""),new String("")};

      if (format == null) {
        data = (String)  fields.get(fieldname);
        return data;
      }
      else {
        if (data.equalsIgnoreCase("")) {
          return "";
        }
        else if (format.indexOf("number") != -1) {
          data = (String) fields.get(fieldname);
          if (data.indexOf(".") != -1) {
            Float tmp = Float.valueOf(data);
            mess[0] = tmp;
          }
          else {
            mess[0] = new Integer(Integer.parseInt(data));
          }
        }
        else if (format.indexOf("date") != -1) {

          data = (String) fields.get(fieldname);

          if (data.indexOf(" ") != -1)
            data = data.substring(0,data.indexOf(" "));


          java.sql.Date dt;
          dt = java.sql.Date.valueOf(data);
          mess[0] = dt;

        }
        else if (format.indexOf("time") != -1) {
          data = (String) fields.get(fieldname);
          if (data.indexOf(".") != -1)
            data = data.substring(0,data.indexOf("."));
          if (data.indexOf(" ") != -1)
            data = data.substring(data.indexOf(" ") + 1).trim();

          java.sql.Time tm;

            tm = java.sql.Time.valueOf(data);
          mess[0] = tm;
        }
        else {
          mess[0] = (String) fields.get(fieldname);
        }

        data = MessageFormat.format("{0," + format + "}",mess);

        return data;
      }
    }


  }

  /**
   * Moves to a specific position within a db, only usable with client cursor
   */
  public boolean moveTo(int location) throws Exception {
    if (this.cursur == DB.CLIENT) {
      if ((location >= 0) && (location < rows.size())) {
        this.cursurLocation = location;
        return true;
      }
      else
        return false;
    }
    else {
      throw new Exception("May only use DB.moveTo(int) when the cursur type is DB.CLIENT");

    }
    //return false;
  }

  public void startWith(int location) throws Exception {
    this.beginWith = location;
  }

  /**
   *pre:rs is populated
   *post:if the there is another record,
   *     advance a record
   *     readin the fields
   *     else return false
   */
  public boolean next() throws Exception {
     if (this.cursur == DB.CLIENT) {
          if (rows.size() > cursurLocation) {
            fields = (Hashtable) rows.elementAt(this.cursurLocation);
            cursurLocation++;
            hasNext = true;
            return true;
          }
          else {
            hasNext = false;
            return false;
          }
        }
        else {//server cursur is default
          if (firstRead) {

              if (rs.next()) {
                hasNext = true;
                readFields();
                return true;
              }
              else {
                hasNext = false;
                return false;
              }

          }
          else {
            firstRead = true;
            return true;
          }
        }


  }

  /**
   * Retruns wether or not the db has another row in it
   */
  public boolean getHasNext() throws Exception {
    if (this.cursur == DB.CLIENT) {
      return (this.cursurLocation < rows.size());
    }
    else {
      return (rs.isLast());
    }
  }
  //****************************************************************************************/
  //*End data retrieval methods                                                            */
  //****************************************************************************************/
}
