/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.testcase.models;

import freemarker.template.*;
import java.io.*;

/**
 * A TemplateTransformModel that includes properties. These properties can be
 * set at model construction time, or, for the purposes of this demonstration,
 * can be passed in from a wrapper TemplateMethodModel.
 */
public class TransformModel1 extends Object implements TemplateTransformModel {

    private boolean m_bAmpersands = false;
    private boolean m_bQuotes = false;
    private boolean m_bTags = false;
    private String  m_aComment = "";

    private static final int READER_BUFFER_SIZE = 4096;

    /** Creates new TransformModel1 */
    public TransformModel1 () {
    }

    /**
     * Indicates whether we escape ampersands. This property can be set either
     * while the model is being constructed, or via a property passed in through
     * a <code>TemplateMethodModel</code>.
     */
    public void setAmpersands( boolean bAmpersands ) {
        m_bAmpersands = bAmpersands;
    }

    /**
     * Indicates whether we escape quotes. This property can be set either
     * while the model is being constructed, or via a property passed in through
     * a <code>TemplateMethodModel</code>.
     */
    public void setQuotes( boolean bQuotes ) {
        m_bQuotes = bQuotes;
    }

    /**
     * Indicates whether we escape tags. This property can be set either
     * while the model is being constructed, or via a property passed in through
     * a <code>TemplateMethodModel</code>.
     */
    public void setTags( boolean bTags ) {
        m_bTags = bTags;
    }

    /**
     * Sets a comment for this transformation. This property can be set either
     * while the model is being constructed, or via a property passed in through
     * a <code>TemplateMethodModel</code>.
     */
    public void setComment( String aComment ) {
        m_aComment = aComment;
    }

    /**
     * Performs a transformation/filter on FreeMarker output.
     *
     * @param source the input to be transformed
     * @param output the destination of the transformation
     */
    public void transform(Reader source,PrintWriter output) throws TemplateModelException {

        // Output the source, converting unsafe certain characters to their
        // equivalent entities.
        int n = 0;
        boolean bCommentSent = false;

        try {
            char[]  aBuffer = new char[ READER_BUFFER_SIZE ];
            int i = source.read( aBuffer );
            while(i >= 0) {
                for( int j = 0; j < i; j++ ) {
                    char c = aBuffer[j];
                    switch (c) {
                        case '&':
                            if( m_bAmpersands ) {
                                output.print("&amp;");
                            } else {
                                output.print( c );
                            }
                            break;
                        case '<':
                            if( m_bTags ) {
                                output.print("&lt;");
                            } else {
                                output.print( c );
                            }
                            break;
                        case '>':
                            if( m_bTags ) {
                                output.print("&gt;");
                            } else {
                                output.print( c );
                            }
                            break;
                        case '"':
                            if( m_bQuotes ) {
                                output.print("&quot;");
                            } else {
                                output.print( c );
                            }
                            break;
                        case '\'':
                            if( m_bQuotes ) {
                                output.print("&apos;");
                            } else {
                                output.print( c );
                            }
                            break;
                        case '*':
                            if( ! bCommentSent ) {
                                output.print( m_aComment );
                                bCommentSent = true;
                            } else {
                                output.print( c );
                            }
                            break;
                        default:
                            output.print(c);
                    }
                    n++;
                }
                i = source.read( aBuffer );
            }
        } catch( IOException e ) {
            throw new TemplateModelException( e.toString() +
                    "...in while loop, position " + Integer.toString( n ) + "." );
        }
    }

    /**
     * @return true if this object is empty.
     */
    public boolean isEmpty() throws TemplateModelException {
        return false;
    }
    
}
