/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.testcase;

import java.io.*;

/**
 * <p>Abstract class used to implement test cases for FreeMarker.  This
 * class is an abstract subclass of the <code>TestCase</code> class
 * provided by <a href="http://junit.sourceforge.net/"
 * target="_top">JUnit</a>.  All FreeMarker testcases are subclassed off
 * this class.
 *
 * <p>This class offers functionality to its subclasses to ease testing where
 * a template is read in, and then written out with a specific data model.
 * Passing the test means that the resulting page matches a given reference
 * text.  This class also supports the creation of those reference texts.<P>
 *
 * <p>To write a new test by subclassing this class, you'll want to do the
 * following:
 *
 * <ul>
 *
 *  <li> Create a template which your test case will use as input, and store
 *       it in the <tt>templates</tt> directory located immediately below 
 *       this directory.  Say it is called <tt>test-foo.html</tt>.</li>
 *
 * <li> Create a subclass of <tt>AbstractTestCase</tt>, and in the
 *       <tt>setUp</tt> method, call <tt>setUpFile("test-foo.html")</tt>.
 *       This will initialize several protected instance variables as
 *       follows: <tt>m_aReferenceText</tt> will contain the contents of
 *       file "reference/test-foo.html", <tt>m_aTemplateText</tt> will
 *       contain the contents of file "templates/test-foo.html", and
 *       <tt>m_aFilename</tt> will contain the string "test-foo.html". 
 *
 *       <p>You can also set up a data model, as per usual JUnit practice.</li>
 *
 *  <li> Now, in the runTest method of your subclass, create a new Template
 *       from a StringReader constructed around <tt>m_aTemplateText</tt>.
 *       Process this with your model, and send the output to a
 *       StringWriter.  Finally, call the <tt>showTestResults</tt> method,
 *       passing in the <tt>m_aReferenceText</tt> and the results of your
 *       test (from your StringWriter object).  This will signal an error
 *       if there is a difference.</li>
 *
 *  <li> In addition, if there is a difference, the actual output produced
 *       by the test will be written out in the current directory (wherever
 *       the test was called from), in this case, as "test-foo.html".
 *       This makes it possible to examine the test result and see what
 *       was actually output.  It also makes it possible to copy that
 *       into the reference/ directory.</li>
 *
 * </ul> 
 */
public abstract class AbstractTestCase extends junit.framework.TestCase {

    private static final int TEST_BUFFER_SIZE = 1024;
    protected String m_aReferenceText;
    protected String m_aTemplateText;
    protected String m_aFilename;

    /**
     * Creates new AbstractTestCase, with a filename from which to populate
     * reference text and template text.
     */
    public AbstractTestCase( String aTestName ) {
        super( aTestName );
    }

    /**
     * Sets up the reference and template files to be used for the test.
     * This would normally be called from the setUp() method in the JUnit
     * framework.
     *
     * @param aFilename the filename to be used for retrieving the reference
     * text and the template
     */
    protected void setUpFiles( String aFilename ) {
        try {
            m_aReferenceText = getReferenceText( aFilename );
            m_aTemplateText = getTemplateText( aFilename );
            m_aFilename = aFilename;
        } catch( IOException e ) {
            throw new RuntimeException( e.getMessage() );
        }
    }
    
    /**
     * Reads text from a file. The file is obtained relative to the 
     * implementing class.
     *
     * @param aFilename the filename to read from
     */
    protected String getTextFromFile( String aFilename ) throws IOException {
        Class thisClass = this.getClass();
        InputStream cStream;
        Reader cReader;
        StringBuffer cBuffer = new StringBuffer();
        char[] aBuffer = new char[ TEST_BUFFER_SIZE ];
        int nLength;

        cStream = thisClass.getResourceAsStream( aFilename );
        
        if( cStream == null ) {
            throw new IOException( "Could not find stream " + aFilename );
        }
        cReader = new InputStreamReader( cStream );

        nLength = cReader.read( aBuffer );

        while( nLength > 0 ) {
            cBuffer.append( aBuffer, 0, nLength );
            nLength = cReader.read( aBuffer );
        }
        return cBuffer.toString();
    }

    /**
     * Gets the reference text for the implementing test class.
     */
    protected String getReferenceText( String aFilename ) throws IOException {

        return getTextFromFile( "reference/" + aFilename );
    }
    
    /**
     * Gets the template text for the implementing test class.
     */
    protected String getTemplateText( String aFilename ) throws IOException {

        return getTextFromFile( "template/" + aFilename );
    }
    
    /**
     * Performs the test on the output text to indicate whether the text is
     * identical to the reference text.
     */
    protected void isTextIdentical( String aReference, String aOutput ) throws TestCaseException {
        int nReferenceLen, nOutputLen, nMinimumLen;
        int nLine, nCharacter, iCurrent;

        nReferenceLen = aReference.length();
        nOutputLen = aOutput.length();

        if( nOutputLen < nReferenceLen ) {
            nMinimumLen = nOutputLen;
        } else {
            nMinimumLen = nReferenceLen;
        }

        nLine = 1;
        nCharacter = 0;

        for( iCurrent = 0; iCurrent < nMinimumLen; iCurrent++ ) {
            if( aReference.charAt( iCurrent ) != aOutput.charAt( iCurrent )) {
                throw new TestCaseException( "Difference encountered at line " +
                    Integer.toString( nLine ) + ", character " +
                    Integer.toString( nCharacter ) + "." );
            }
            nCharacter++;
            
            if( aReference.charAt( iCurrent ) == 10 ) {
                nLine++;
                nCharacter = 0;
            }
        }
        if( nOutputLen > nReferenceLen ) {
            throw new TestCaseException( "Output text is " + 
                ( Integer.toString( nOutputLen - nReferenceLen )) +
                " characters longer than reference text." );
        } else if( nReferenceLen > nOutputLen ) {
            throw new TestCaseException( "Output text is " + 
                ( Integer.toString( nReferenceLen - nOutputLen )) +
                " characters shorter than reference text." );
        }
    }

    /**
     * Verify that the output of a test is identical to the reference text.
     * If they are identical, say nothing (in the JUnit tradition).  If
     * they differ, output the first line number and character where they
     * differ.
     *
     * @param aReference the reference text
     * @param aOutput the output from the test
     */
    protected void showTestResults( String aReference, String aOutput ) {

        try {
            isTextIdentical( m_aReferenceText, aOutput );
        } catch( TestCaseException error ) {
            try {
                writeText( m_aFilename, aOutput );
            } catch( IOException writeException ) {
                fail( writeException.getMessage() );
            }
            assertTrue( error.getMessage(), false );
        }
    }

    /**
     * Writes text to a given filename. Useful when we want to set up the
     * reference file for future tests to compare against.
     */
    protected void writeText( String aFilename, String aText ) throws IOException {
        FileOutputStream cStream = new FileOutputStream( aFilename );
        OutputStreamWriter cWriter = new OutputStreamWriter( cStream );

        cWriter.write( aText );
        cWriter.flush();
        cWriter.close();
        cStream.close();
    }

}
