/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.utility;

import freemarker.template.*;
import java.io.*;

/**
 * A re-implementation of the &lt;compress&gt; template instruction as a 
 * <code>TemplateTransformModel</code>. This implementation behaves exactly
 * the same way as the compress tag.
 *
 * <p>Usage:<br />
 * From java:</p>
 * <pre>
 * TemplateModelRoot root = new SimpleHash();
 *
 * root.put( "compressWhitespace", new freemarker.template.utility.LegacyCompress() );
 *
 * ...
 * </pre>
 *
 * <p>From your FreeMarker template:</p>
 * <pre>
 * 
 * The following is executed:
 * &lt;transform compressWhitespace&gt;
 *   &lt;p&gt;This paragraph has all whitespace removed.&lt;/p&gt;
 * &lt;/transform&gt;
 *
 * ...
 * </pre>
 *
 * @version $Id: LegacyCompress.java,v 1.2 2001/05/29 11:54:10 run2000 Exp $
 * @see freemarker.template.utility.CompressWhitespace
 */
public class LegacyCompress extends Object implements TemplateTransformModel {

    private static final int READER_BUFFER_SIZE = 4096;

    /** Creates new LegacyCompress */
    public LegacyCompress() {
    }

    /**
     * Re-implementation of the &lt;compress&gt; FreeMarker tag.
     *
     * @param source the input to be transformed
     * @param output the destination of the transformation
     */
    public void transform(Reader source, PrintWriter output) throws TemplateModelException {

        // Output the string, stripping non-whitespace characters outside
        // of tags.
        int n = 0;
		boolean inTag = false;

        try {
            char[]  aBuffer = new char[ READER_BUFFER_SIZE ];
            int i = source.read( aBuffer );
            while(i >= 0) {
                for( int j = 0; j < i; j++ ) {
                    char c = aBuffer[j];
                    switch (c) {
                        case '<':
                            inTag = true;
                            output.print(c);
                            break;
                        case '>':
                            inTag = false;
                            output.print(c);
                            break;
                        default:
                            if (inTag) {
                                output.print(c);
                            } else {
                                if (!Character.isWhitespace(c)) {
                                    output.print(c);
                                }
                            }
                    }

                    n++;
                }
                i = source.read( aBuffer );
            }
        } catch( IOException e ) {
            throw new TemplateModelException( e.toString() +
                    "...in LegacyCompress, position " + Integer.toString( n ) + "." );
        }
    }

    /**
     * @return true if this object is empty.
     */
    public boolean isEmpty() throws TemplateModelException {
        return false;
    }
}
