/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.utility;

import freemarker.template.*;
import java.io.*;

/**
 * Performs an HTML escape of a given template fragment. Specifically,
 * &lt; &gt; &quot; and &amp; are all turned into entities.
 *
 * <p>Usage:<br />
 * From java:</p>
 * <pre>
 * TemplateModelRoot root = new SimpleHash();
 *
 * root.put( "htmlEscape", new freemarker.template.utility.HtmlEscape() );
 *
 * ...
 * </pre>
 *
 * <p>From your FreeMarker template:</p>
 * <pre>
 * 
 * The following is executed:
 * &lt;transform htmlEscape&gt;
 *   &lt;p&gt;This paragraph has all HTML special characters escaped.&lt;/p&gt;
 * &lt;/transform&gt;
 *
 * ...
 * </pre>
 *
 * @version $Id: HtmlEscape.java,v 1.3 2001/05/29 11:54:10 run2000 Exp $
 * @see freemarker.template.utility.XmlEscape
 */
public class HtmlEscape extends Object implements TemplateTransformModel {

    private static final int READER_BUFFER_SIZE = 4096;

    /** Creates new HtmlEscape */
    public HtmlEscape() {
    }

    /**
     * Transforms an HTML-unescaped stream into HTML-escaped form. This means:
     * &lt; &gt; &amp; and &quot; are all escaped into their equivalent entities.
     *
     * @param source the input to be transformed
     * @param output the destination of the transformation
     */
    public void transform(Reader source, PrintWriter output) throws TemplateModelException {

        // Output the source, converting unsafe HTML characters to their
        // equivalent entities.
        int n = 0;
        try {
            char[]  aBuffer = new char[ READER_BUFFER_SIZE ];
            int i = source.read( aBuffer );
            while(i >= 0) {
                for( int j = 0; j < i; j++ ) {
                    char c = aBuffer[j];
                    switch (c) {
                        case '&':
                            output.print("&amp;");
                            break;
                        case '<':
                            output.print("&lt;");
                            break;
                        case '>':
                            output.print("&gt;");
                            break;
                        case '"':
                            output.print("&quot;");
                            break;
                        default:
                            output.print(c);
                    }
                    n++;
                }
                i = source.read( aBuffer );
            }
        } catch( IOException e ) {
            throw new TemplateModelException( e.toString() +
                    "...in HtmlEscape, position " + Integer.toString( n ) + "." );
        }
    }

    /**
     * @return true if this object is empty.
     */
    public boolean isEmpty() throws TemplateModelException {
        return false;
    }
}
