/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.utility;

import freemarker.template.*;
import java.io.*;
import java.util.*;

/**
 * <p>Gives FreeMarker the the ability to execute external commands. Will fork 
 * a process, and inline anything that process sends to stdout in the template. 
 * Based on a patch submitted by Peter Molettiere.</p>
 *
 * <p>BE CAREFUL! this tag, depending on use, may allow you
 * to set something up so that users of your web
 * application could run arbitrary code on your server.
 * This can only happen if you allow unchecked GET/POST
 * submissions to be used as the command string in the
 * exec tag.</p>
 *
 * <p>Usage:<br />
 * From java:</p>
 * <pre>
 * TemplateModelRoot root = new SimpleHash();
 *
 * root.put( "exec", new freemarker.template.utility.Execute() );
 *
 * ...
 * </pre>
 *
 * <p>From your FreeMarker template:</p>
 * <pre>
 * 
 * The following is executed:
 * ${exec( "/usr/bin/ls" )}
 * 
 * ...
 * </pre>
 *
 * @version $Id: Execute.java,v 1.1.1.1 2001/05/10 10:05:19 run2000 Exp $
 */
public class Execute implements freemarker.template.TemplateMethodModel {

    private final static int OUTPUT_BUFFER_SIZE = 1024;

    /** Creates new Execute FreeMarker template */
    public Execute () {
    }

    /**
     * Executes a method call.
     *
     * @param arguments a <tt>List</tt> of <tt>String</tt> objects containing the values
     * of the arguments passed to the method.
     * @return the <tt>TemplateModel</tt> produced by the method, or null.
     */
    public TemplateModel exec (List arguments) throws TemplateModelException {
        String aExecute;
        StringBuffer    aOutputBuffer = new StringBuffer();
        TemplateModel   cModel;

        if( arguments.size() < 1 ) {
            throw new TemplateModelException( "Need an argument to execute" );
        }

        aExecute = (String)(arguments.get(0));

		try {
		    Process exec = Runtime.getRuntime().exec( aExecute );

		    // stdout from the process comes in here
		    InputStream execOut = exec.getInputStream();
            Reader execReader = new InputStreamReader( execOut );

		    char[] buffer = new char[ OUTPUT_BUFFER_SIZE ];
		    int bytes_read = execReader.read( buffer );

		    while( bytes_read > 0 ) {
                aOutputBuffer.append( buffer, 0, bytes_read );
                bytes_read = execReader.read( buffer );
		    }
		}
		catch( IOException ioe ) {
		    throw new TemplateModelException( ioe.getMessage() );
		}
        finally {
            cModel = new SimpleScalar( aOutputBuffer.toString() );
            return cModel;
        }
    }

    /**
     * @return true if this object is empty.
     */
    public boolean isEmpty () throws TemplateModelException {
        return false;
    }
    
}
