/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.utility;

import freemarker.template.*;
import java.io.*;

/**
 * A transform model that compresses each occurrence of consecutive whitespace
 * down to a single space character. Leading and trailing whitespace is also
 * removed.
 *
 * <p>Usage:<br />
 * From java:</p>
 * <pre>
 * TemplateModelRoot root = new SimpleHash();
 *
 * root.put( "compressWhitespace", new freemarker.template.utility.CompressWhitespace() );
 *
 * ...
 * </pre>
 *
 * <p>From your FreeMarker template:</p>
 * <pre>
 * 
 * The following is executed:
 * &lt;transform compressWhitespace&gt;
 *   &lt;p&gt;This paragraph has all whitespace reduced to a single 
 *   space character.&lt;/p&gt;
 * &lt;/transform&gt;
 *
 * ...
 * </pre>
 *
 * @version $Id: CompressWhitespace.java,v 1.3 2001/06/12 10:11:24 run2000 Exp $
 * @see freemarker.template.utility.LegacyCompress
 */
public class CompressWhitespace extends Object implements TemplateTransformModel {

    private static final int READER_BUFFER_SIZE = 4096;

    /** Creates new CompressWhitespace */
    public CompressWhitespace () {
    }

    /**
     * Compresses whitespace within the marked portion of a FreeMarker template.
     *
     * @param source the input to be transformed
     * @param output the destination of the transformation
     */
    public void transform(Reader source, PrintWriter output) throws TemplateModelException {

        // Output the string, stripping non-whitespace characters outside
        // of tags.
        int n = 0;
		boolean whitespaceSeen = false;
        boolean atBeginning = true;

        try {
            char[]  aBuffer = new char[ READER_BUFFER_SIZE ];
            int i = source.read( aBuffer );
            while(i >= 0) {
                for( int j = 0; j < i; j++ ) {
                    char c = aBuffer[j];

                    if( Character.isWhitespace(c) ) {
                        whitespaceSeen = true;
                    } else {
                        if( atBeginning ) {
                            atBeginning = false;
                            whitespaceSeen = false;
                        } else if( whitespaceSeen ) {
                            output.print(' ');
                            whitespaceSeen = false;
                        }
                        output.print(c);
                    }
                    n++;
                }
                i = source.read( aBuffer );
            }
        } catch( IOException e ) {
            throw new TemplateModelException( e.toString() +
                    "...in CompressWhitespace, position " + Integer.toString( n ) + "." );
        }
    }

    /**
     * @return true if this object is empty.
     */
    public boolean isEmpty() throws TemplateModelException {
        return false;
    }
}
