/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.instruction;

import freemarker.template.*;
import freemarker.template.expression.*;
import java.io.*;

/**
 * An instruction that processes a TemplateTransformModel.
 */
public class TransformInstruction extends GenericStartInstruction {

    protected   Variable transformVariable;

    /** Creates new TransformInstruction */
    public TransformInstruction() {
    }

    /** Creates new TransformInstruction, with a given transformation
     * @param variable the variable representing the transformation to use
     */
    public TransformInstruction( Variable variable ) {
        transformVariable = variable;
    }

    /**
     * @param listVariable a variable referring to a TemplateTransformModel.
     */
    public void setTransformTemplate( Variable variable ) {
        transformVariable = variable;
    }

    /**
     * @return a string describing this <tt>ContainerInstruction</tt>.
     */
    public String getTypeName() {
        return "transform";
    }
    
    /**
     * <p>Processes the contents of this TemplateProcessor and outputs the
     * resulting text to a PrintWriter.</p>
     *
     * <p>Note:
     * This implementation is time-efficient rather than space efficient.
     * For a space-efficient tradeoff, use <code>PipeReader</code> and 
     * <code>PipeWriter</code> instead. PipeWriter would need to be in
     * a separate thread.</p>
     *
     * @param modelRoot the root node of the data model.
     * @param out a PrintWriter to send the output to.
     */
    public void process(TemplateModelRoot modelRoot, PrintWriter out) {

        // Make sure the TransformTemplate exists
		if (transformVariable == null) {
			out.print(Template.formatErrorMessage("Transformation variable undefined."));
			return;
		}

        // Process the body into a string buffer.
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);

        // Get the content we have so far
        body.process(modelRoot, pw);
		pw.close();

		String transformVariableName;
		try {
			transformVariableName = transformVariable.getName(modelRoot);
		} catch (TemplateException te) {
			out.print(Template.formatErrorMessage("Couldn't resolve name of transform variable:\n" +
				te.getMessage()));
			return;
		}

		TemplateModel baseModel;
		try {
			baseModel = transformVariable.getAsTemplateModel(modelRoot);
		} catch (TemplateException te) {
			out.print(Template.formatErrorMessage("Couldn't get referent of " +
				transformVariableName +
				":\n" +
				te.getMessage()));
			return;
		}

		if (baseModel == null) {
			return;
		}

		if (baseModel instanceof TemplateTransformModel) {
            // Now read from the string buffer
            Reader inData = new StringReader( sw.toString() );

			try {
                ((TemplateTransformModel)baseModel).transform( inData, out );

            } catch (TemplateModelException e) {
				out.print(Template.formatErrorMessage("Couldn't perform transformation " +
					transformVariableName + ":\n" +
					Template.getStackTrace( e )));
			}
        } else {
			out.print(Template.formatErrorMessage(transformVariableName +
				" is not a TemplateTransformModel."));
        }
    }
    
    /**
     * @return true if the <tt>EndInstruction</tt> is the right kind
     * for this <tt>ContainerInstruction</tt>.
     */
    public boolean testEndInstruction(Instruction endInstruction) {
		return (endInstruction.getEndType() == TRANSFORM_END);
    }
    
}
