/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.instruction;

import freemarker.template.*;
import freemarker.template.compiler.*;
import freemarker.template.expression.*;
import java.io.*;
import java.util.*;

/**
 * An instruction representing a switch-case structure.
 */
public class SwitchInstruction extends StartInstruction {

    private List caseInstructions = new LinkedList();
	private CaseInstruction defaultCase;
	private Expression testExpression;

    public SwitchInstruction() { }

    /**
     * @param testExpression the expression to be tested.
     */
    public SwitchInstruction(Expression testExpression) {
		setTestExpression(testExpression);
    }

    /**
     * @param condition the condition for the if statement.
     */
    public void setTestExpression(Expression testExpression) {
		this.testExpression = testExpression;
    }

    /**
     * @param caseInstruction a CaseInstruction.
     */
    public void addCase(CaseInstruction caseInstruction) {
		caseInstructions.add(caseInstruction);
		if (caseInstruction.isDefault()) {
			defaultCase = caseInstruction;
		}
    }

	public void callBuilder(TemplateBuilder builder,
		RuntimeStructuralElement element) throws TemplateException {
		builder.buildStatement(this, element);
	}

    /**
     * @param modelRoot the root node of the data model.
     * @param out a PrintWriter to send the output to.
     */
    public void process(TemplateModelRoot modelRoot, PrintWriter out) {
		Equals equalsOp = new Equals();
		try {
			equalsOp.setLeft(testExpression);
		} catch (TemplateException e) { }
		
		boolean processedCase = false;
		Iterator iterator = caseInstructions.iterator();
		while (iterator.hasNext()) {
			CaseInstruction caseInstruction = (CaseInstruction)iterator.next();
			boolean processCase = false;

			// Fall through if a previous case tested true.
			if (processedCase) {
				processCase = true;
			} else if (!caseInstruction.isDefault()) {
				// Otherwise, if this case isn't the default, test it.
				Expression caseExpression = caseInstruction.getExpression();
				try {
					equalsOp.setRight(caseExpression);
					processCase = equalsOp.isTrue(modelRoot);
				} catch (TemplateException e) {
					out.print(Template.formatErrorMessage(e.getMessage()));
				}
			}

			if (processCase) {
				caseInstruction.process(modelRoot, out);
				processedCase = true;
				if (caseInstruction.hasBreak()) {
					break;
				}
			}
		}

		// If we didn't process any cases, and we have a default,
		// process it.
		if (!processedCase && defaultCase != null) {
			defaultCase.process(modelRoot, out);
		}
    }
}
