/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.instruction;

import freemarker.template.*;
import freemarker.template.expression.*;
import java.io.*;

/**
 * An instruction that processes a TemplateListModel.
 */
public class ListInstruction extends GenericStartInstruction {

    private Variable listVariable;
    private Identifier indexVariable;

    public ListInstruction() { }

    /**
     * @param listVariable a variable referring to a TemplateListModel.
     * @param indexVariable an arbitrary index variable.
     */
    public ListInstruction(Variable listVariable, Identifier indexVariable) {
		setListVariable(listVariable);
		setIndexVariable(indexVariable);
    }

    /**
     * @param listVariable a variable referring to a TemplateListModel.
     */
    public void setListVariable(Variable listVariable) {
		this.listVariable = listVariable;
    }

    /**
     * @param indexVariable an arbitrary index variable.
     */
    public void setIndexVariable(Identifier indexVariable) {
		this.indexVariable = indexVariable;
    }

	public boolean testEndInstruction(Instruction endInstruction) {
		return (endInstruction.getEndType() == LIST_END);
	}

	public String getTypeName() {
		return "list";
	}

    /**
     * Processes the list.
     *
     * @param modelRoot the root node of the data model.
     * @param out a PrintWriter to send the output to.
     */
    public void process(TemplateModelRoot modelRoot, PrintWriter out) {
		if (listVariable == null) {
			out.print(Template.formatErrorMessage("List variable undefined."));
			return;
		}
		if (indexVariable == null) {
			out.print(Template.formatErrorMessage("Index variable undefined."));
			return;
		}
		if (body == null) {
			return;
		}

		String listVariableName;
		try {
			listVariableName = listVariable.getName(modelRoot);
		} catch (TemplateException te) {
			out.print(Template.formatErrorMessage("Couldn't resolve name of list variable:\n" +
				te.getMessage()));
			return;
		}

		TemplateModel baseModel;
		try {
			baseModel = listVariable.getAsTemplateModel(modelRoot);
		} catch (TemplateException te) {
			out.print(Template.formatErrorMessage("Couldn't get referent of " +
				listVariableName +
				":\n" +
				te.getMessage()));
			return;
		}

		if (baseModel == null) {
			return;
		}

		String indexName = null;
		try {
			indexName = indexVariable.getName(modelRoot);
		} catch (TemplateException te) { }

		// Hide the old value of the loop index variable.
		TemplateModel oldIndexReferent = null;
		try {
			oldIndexReferent = indexVariable.getAsTemplateModel(modelRoot);
		} catch (TemplateException te) { }

		if (baseModel instanceof TemplateListModel) {
			TemplateListModel baseListModel = (TemplateListModel)baseModel;

			// Loop, setting the index to refer to each element in turn.
			TemplateModel currElement;
			try {
				if (!baseListModel.isRewound()) {
					baseListModel.rewind();
				}
				while (baseListModel.hasNext()) {
					currElement = baseListModel.next();
					modelRoot.put(indexName, currElement);
					body.process(modelRoot, out);
				}
			} catch (TemplateModelException e) {
				out.print(Template.formatErrorMessage("Couldn't read from list " +
					listVariableName + ":\n" +
					Template.getStackTrace(e)));
			}
		} else if (baseModel instanceof TemplateScalarModel) {
			// Treat a scalar as a list with one element.
			modelRoot.put(indexName, baseModel);
			body.process(modelRoot, out);
		} else {
			out.print(Template.formatErrorMessage(listVariableName +
				" is not a TemplateListModel or a TemplateScalarModel."));
		}

		// Restore the old value of the loop index variable, if there was one.
		if (oldIndexReferent != null) {
			modelRoot.put(indexName, oldIndexReferent);
		} else {
			modelRoot.remove(indexName);
		}
    }
}
