/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.instruction;

import freemarker.template.*;
import freemarker.template.expression.*;
import java.io.*;
import java.util.*;

/**
 * An instruction that gets another template from a <tt>TemplateCache</tt>, and processes
 * it within the current template.
 */
public class IncludeInstruction extends EmptyInstruction {

    private Template template;
	private Expression templateName;

    public IncludeInstruction() { }

    /**
     * @param variable the template that this <tt>Include</tt> is a part of.
 	 * @param templateName the name, in the <tt>TemplateCache</tt>, of the template to be included.
     */
    public IncludeInstruction(Template template, Expression templateName) {
		setTemplate(template);
		setIncludedTemplateName(templateName);
    }

    /**
     * @param variable the template that this <tt>Include</tt> is a part of.
     */
    public void setTemplate(Template template) {
		this.template = template;
    }

	/**
 	 * @param templateName the name, in the <tt>TemplateCache</tt>, of the template to be included.
	 */
	public void setIncludedTemplateName(Expression templateName) {
		this.templateName = templateName;
	}

    /**
     * @param modelRoot the root node of the data model.
     * @param out a <tt>PrintWriter</tt> to send the output to.
     */
    public void process(TemplateModelRoot modelRoot, PrintWriter out) {
		if (template == null) {
			out.print(Template.formatErrorMessage("The TemplateInclude object wasn't given a reference to the current template."));
			return;
		}

		TemplateCache cache = template.getTemplateCache();
		if (cache == null) {
			out.print(Template.formatErrorMessage("The current template wasn't given a reference to a TemplateCache."));
			return;
		}

		String templateNameString = null;
		try {
			templateNameString = templateName.getValue(modelRoot);
		} catch (TemplateException e) {
			out.print(Template.formatErrorMessage("Error in template name expression: " +
				Template.getStackTrace(e)));
			return;
		}

		Template includedTemplate = cache.getTemplate(templateNameString);
		if (includedTemplate == null) {
			out.print(Template.formatErrorMessage("Template \"" +
				templateNameString + "\" not found in cache."));
			return;
		}

		// Copy the functions from both templates into the data model.
		copyFunctions(template, modelRoot);
		copyFunctions(includedTemplate, modelRoot);

		includedTemplate.process(modelRoot, out);
    }

	/**
	 * Copies functions from a template into a data model.
	 */
	private void copyFunctions(Template template, TemplateModelRoot modelRoot) {
		Set functionNames = template.getFunctionNames();
		Iterator iterator = functionNames.iterator();
		while (iterator.hasNext()) {
			String functionName = (String)iterator.next();
			FunctionInstruction function = template.getFunction(functionName);
			modelRoot.put(functionName, new FunctionModel(function));
		}
	}
}
