/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.instruction;

import freemarker.template.*;
import freemarker.template.compiler.*;
import freemarker.template.expression.*;
import java.io.*;

/**
 * An instruction representing an if-else structure.  The "if" part of
 * the structure will be executed if the condition expression
 * evaluates to a non-null value.
 */
public class IfInstruction extends StartInstruction {

    private Expression condition;
    private TemplateProcessor ifBlock;
    private TemplateProcessor elseBlock;

    public IfInstruction() { }

    /**
     * @param condition the condition for the if statement.
     */
    public IfInstruction(Expression condition) {
		setCondition(condition);
    }

    /**
     * @param condition the condition for the if statement.
     */
    public void setCondition(Expression condition) {
		this.condition = condition;
    }

    /**
     * @param ifBlock a TemplateProcessor to be processed if the condition is true.
     */
    public void setIfBlock(TemplateProcessor ifBlock) {
		this.ifBlock = ifBlock;
    }

    /**
     * @param elseBlock a TemplateProcessor to be processed if the condition is false.
     */
    public void setElseBlock(TemplateProcessor elseBlock) {
		this.elseBlock = elseBlock;
    }

	public void callBuilder(TemplateBuilder builder,
		RuntimeStructuralElement element) throws TemplateException {
		builder.buildStatement(this, element);
	}

    /**
     * @param modelRoot the root node of the data model.
     * @param out a PrintWriter to send the output to.
     */
    public void process(TemplateModelRoot modelRoot, PrintWriter out) {
		if (condition == null) {
			out.print(Template.formatErrorMessage("Condition undefined."));
			return;
		}

		boolean isTrue;
		try {
			isTrue = condition.isTrue(modelRoot);
		} catch (TemplateException te) {
			out.print(Template.formatErrorMessage(te.getMessage()));
			return;
		}

		if (isTrue) {
			if (ifBlock != null) {
				ifBlock.process(modelRoot, out);
			}
		} else {
			if (elseBlock != null) {
				elseBlock.process(modelRoot, out);
			}
		}
    }
}
