/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.instruction;

import freemarker.template.*;
import java.io.*;

/**
 * An instruction that removes whitespace outside of HTML tags.
 */
public class CompressInstruction extends GenericStartInstruction {

	public CompressInstruction() { }

	public boolean testEndInstruction(Instruction endInstruction) {
		return (endInstruction.getEndType() == COMPRESS_END);
	}

	public String getTypeName() {
		return "compress";
	}

    /**
     * @param modelRoot the root node of the data model.
     * @param out a PrintWriter to send the output to.
     */
    public void process(TemplateModelRoot modelRoot, PrintWriter out) {
		// Process the body into a string.
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);
		body.process(modelRoot, pw);
		pw.close();
		String outString = sw.toString();

		// Output the string, stripping non-whitespace characters outside
		// of tags.
		boolean inTag = false;
		for (int i = 0; i < outString.length(); i++) {
			char c = outString.charAt(i);
			switch (c) {
				case '<':
					inTag = true;
					out.print(c);
					break;
				case '>':
					inTag = false;
					out.print(c);
					break;
				default:
					if (inTag) {
						out.print(c);
					} else {
						if (!Character.isWhitespace(c)) {
							out.print(c);
						}
					}
			}
		}
	}
}
