/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.instruction;

import freemarker.template.*;
import freemarker.template.expression.*;
import java.io.*;

/**
 * Represents a case in a switch statement.
 */
public class CaseInstruction extends EndInstruction implements TemplateProcessor {
	private Expression expression;
	private boolean hasBreak;
	private boolean isDefault;
	private TemplateProcessor body;

	public CaseInstruction() { }

	/**
	 * @param expression the <tt>Expression</tt> associated with this <tt>Case</tt>.
	 */
	public CaseInstruction(Expression expression) {
		setExpression(expression);
	}

	/**
	 * @param expression the <tt>Expression</tt> associated with this <tt>Case</tt>.
	 */
	public void setExpression(Expression expression) {
		this.expression = expression;
	}

	/**
	 * @return the <tt>Expression</tt> associated with this <tt>TemplateCase</tt>.
	 */
	public Expression getExpression() {
		return expression;
	}

	/**
	 * @param isDefault true if this is the default case.
	 */
	public void setIsDefault(boolean isDefault) {
		this.isDefault = isDefault;
	}

	/**
	 * @return true if this is the default case.
	 */
	public boolean isDefault() {
		return isDefault;
	}

	/**
	 * @param body the <tt>TemplateProcessor</tt> to be processed if this <tt>Case</tt> passes the test.
	 */
	public void setBody(TemplateProcessor body) {
		this.body = body;
	}

	/**
	 * @param break true if this <tt>TemplateCase</tt> has a <tt>break</tt>.
	 */
	public void setHasBreak(boolean hasBreak) {
		this.hasBreak = hasBreak;
	}

	/**
	 * @return true if this <tt>Case</tt> has a <tt>break</tt>.
	 */
	public boolean hasBreak() {
		return hasBreak;
	}

    /**
     * @param modelRoot the root node of the data model.
     * @param out a PrintWriter to send the output to.
     */
    public void process(TemplateModelRoot modelRoot, PrintWriter out) {
		if (body != null) {
			body.process(modelRoot, out);
		}
	}
	
	public int getEndType() {
		return CASE;
	}
}
