/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.instruction;

import freemarker.template.*;
import freemarker.template.expression.*;
import java.io.*;
import java.util.*;

/**
 * An instruction representing a function call.
 */
public class CallInstruction extends EmptyInstruction {
	private Template template;
	private String functionName;
	private List arguments; // Expression objects.

    public CallInstruction(Template template, String functionName, List arguments) {
		this.template = template;
		this.functionName = functionName;
		this.arguments = arguments;
	}

    /**
     * @param modelRoot the root node of the data model.
     * @param out a PrintWriter to send the output to.
     */
    public void process(TemplateModelRoot modelRoot, PrintWriter out) {
		// Look for the function first in the template, then in the data model,
		// where it might have been put by an IncludeInstruction.
		FunctionInstruction function = template.getFunction(functionName);
		if (function == null) {
			TemplateModel model = null;
			try {
				model = modelRoot.get(functionName);
			} catch (TemplateModelException e) {
				out.print(Template.formatErrorMessage("Couldn't get function " +
					functionName +
					":\n" +
					Template.getStackTrace(e)));
				return;
			}

			if (model instanceof FunctionModel) {
				function = ((FunctionModel)model).getFunction();
			}
		}

		if (function == null) {
			out.print(Template.formatErrorMessage("Function " + functionName +
				" has not been defined."));
			return;
		}

		List argumentNames = function.getArgumentNames();
		if (argumentNames.size() != arguments.size()) {
			out.print(Template.formatErrorMessage("Wrong number of arguments in function call."));
			return;
		}

		// Put the arguments in the data model under the argument names,
		// temporarily hiding the current values of those names.
		Map oldValues = new HashMap();
		Iterator argNamesIterator = argumentNames.iterator();
		Iterator argIterator = arguments.iterator();
		while (argIterator.hasNext()) {
			String argName = (String)argNamesIterator.next();

			// Hide the current value of whatever has this name.
			TemplateModel oldReferent = null;
			try {
				oldReferent = modelRoot.get(argName);
			} catch (TemplateModelException te) { }
			if (oldReferent != null) {
				oldValues.put(argName, oldReferent);
			}

			// Put the argument's value in the data model.
			Expression argExpression = (Expression)argIterator.next();
			TemplateModel argModel = null;
			try {
				argModel = argExpression.getAsTemplateModel(modelRoot);
			} catch (TemplateException e) {
				out.print(Template.formatErrorMessage(e.getMessage()));
				return;
			}
			modelRoot.put(argName, argModel);
		}

		// Call the function.
		function.process(modelRoot, out);

		// Restore the old values of the argument names.
		argNamesIterator = argumentNames.iterator();
		while (argNamesIterator.hasNext()) {
			String argName = (String)argNamesIterator.next();
			TemplateModel oldReferent = (TemplateModel)oldValues.get(argName);
			if (oldReferent != null) {
				modelRoot.put(argName, oldReferent);
			} else {
				modelRoot.remove(argName);
			}
		}
    }
}
