/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.instruction;

import freemarker.template.*;
import freemarker.template.expression.*;
import java.io.*;

/**
 * An instruction that assigns a literal or reference, to a single-identifier variable.
 */
public class AssignInstruction extends EmptyInstruction {

    private Identifier variable;
    private Expression value;

    public AssignInstruction() { }

    /**
     * @param variable the variable to assign to.
	 * @param value the expression to assign.
     */
	public AssignInstruction(Identifier variable, Expression value) {
		setVariable(variable);
		setValue(value);
    }

    /**
     * @param variable the variable to assign to.
     */
    public void setVariable(Identifier variable) {
		this.variable = variable;
    }

    /**
	 * @param value the expression to assign.
     */
    public void setValue(Expression value) {
		this.value = value;
    }
	
    /**
     * @param modelRoot the root node of the data model.
     * @param out a PrintWriter to send the output to.
     */
    public void process(TemplateModelRoot modelRoot, PrintWriter out) {
		if (variable == null) {
			out.print(Template.formatErrorMessage("Assignment variable undefined."));
			return;
		}

		if (value == null) {
			out.print(Template.formatErrorMessage("Assignment value undefined."));
			return;
		}

		try {
			String variableName = variable.getName(modelRoot);
			modelRoot.put(variableName, value.getAsTemplateModel(modelRoot));
		} catch (TemplateException e) {
			out.print(Template.formatErrorMessage("Couldn't perform assignment:\n" +
				e.getMessage()));
			return;
		}
    }
}
