/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.expression;

import freemarker.template.*;
import freemarker.template.instruction.*;
import java.util.*;
import java.io.*;

/**
 * An abstract class for expressions that get their values from the data model.
 */
public abstract class Variable implements Expression {
    public abstract String getName(TemplateModelRoot modelRoot) throws TemplateException;

    public String getValue(TemplateModelRoot modelRoot) throws TemplateException {
	String stringValue;
	TemplateModel referentModel;

	try {
	    referentModel = getAsTemplateModel(modelRoot);
	} catch (TemplateException e) {
	    throw new TemplateException("Couldn't get referent of " +
					getName(modelRoot) + ":\n" + e.getMessage());
	}

	if (referentModel == null) {
	    return null;
	}

	if (!(referentModel instanceof TemplateScalarModel)) {
	    throw new TemplateException(getName(modelRoot) +
					" is not a TemplateScalarModel, it's a " +
					referentModel.getClass().getName() + ".");
	}

	try {
	    if (referentModel.isEmpty()) {
		return null;
	    }
	} catch (TemplateModelException e) {
	    throw makeReadException(modelRoot, e);
	}

	TemplateScalarModel referentScalarModel = (TemplateScalarModel)referentModel;
	try {
	    stringValue = referentScalarModel.getAsString();
	} catch (TemplateModelException e) {
	    throw new TemplateException("Couldn't read string value of " +
					getName(modelRoot) + ":\n" +
					Template.getStackTrace(e));
	}
	return stringValue;
    }

    public boolean isTrue(TemplateModelRoot modelRoot) throws TemplateException {
	TemplateModel referent = getAsTemplateModel(modelRoot);
	try {
	    return !(referent == null || referent.isEmpty());
	} catch (TemplateModelException e) {
	    throw makeReadException(modelRoot, e);
	}
    }

    private TemplateException makeReadException(TemplateModelRoot modelRoot,
						Exception e) throws TemplateException {
	return new TemplateException("Couldn't read from " +
				     getName(modelRoot) + ":\n" + e.getMessage());
    }
}
