/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 *
 * 22 October 1999: This class added by Holger Arendt.
 */

package freemarker.template.expression;

import freemarker.template.*;
import java.util.*;

/**
 * A unary operator that calls a TemplateMethodModel.  It associates with the
 * <tt>Identifier</tt> or <tt>Dot</tt> to its left.
 */
public class MethodCall extends Variable implements Unary {

    private ListLiteral arguments;

    private Variable target;

    public MethodCall() { }

    public MethodCall(List arguments) {
		setArguments(arguments);
    }

    public void setArguments(List arguments) {
        this.arguments = new ListLiteral(arguments);
    }

    public void setTarget(Expression target) throws TemplateException {
        if (!(target instanceof Variable && target.isComplete())) {
            throw new TemplateException("Syntax error in expression.");
        }
        this.target = (Variable)target;
    }

    public Expression getTarget() {
        return target;
    }

    public boolean isComplete() {
        return !(arguments == null || target == null);
    }

    public int getAssociationType() {
        return POSTFIX;
    }

    public TemplateModel getAsTemplateModel(TemplateModelRoot modelRoot) throws TemplateException {
        TemplateModel targetModel = target.getAsTemplateModel(modelRoot);
        if (targetModel == null) return null;

        if (!(targetModel instanceof TemplateMethodModel)) {
            throw new TemplateException(target.getName(modelRoot) +
                                        " is not a TemplateMethodModel, it's a " +
										targetModel.getClass().getName() + ".");
        }

        TemplateMethodModel targetMethod = (TemplateMethodModel)targetModel;
		List argumentStrings = arguments.getValueList(modelRoot);

		try {
			return targetMethod.exec(argumentStrings);
		} catch (TemplateModelException e) {
			throw new TemplateException("Error calling method in expression:\n" +
										Template.getStackTrace(e));
		}
    }
	
	public String getName( TemplateModelRoot modelRoot ) throws TemplateException {
		return target.getName(modelRoot);
	}
}
