/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * 28 June 1999: Modified by Steve Chiu to ignore extra parentheses.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.expression;

import freemarker.template.*;
import java.util.*;

/**
 * A TemplateParser can use this class's static <tt>build()</tt>
 * method to build a complete <tt>Expression</tt> from a
 * <tt>List</tt> of <tt>ExpressionElement</tt>s.
 */
public class ExpressionBuilder {

    // An array of arrays of operator classes, arranged in descending
	// order of precendence.
    private static final Class [][] opClasses = {
		{ Dot.class, MethodCall.class, DynamicKeyName.class },
		{ Not.class },
		{ Concatenate.class },
		{ Equals.class,
		  NotEquals.class },
		{ And.class },
		{ Or.class }
	};
    private static final int numLevels = opClasses.length;

    // Error messages.
    private static final String MISMATCHED_PARENS = "Mismatched parentheses.";
    private static final String EMPTY_PARENS = "Empty parentheses.";
    private static final String MISSING_LEFT = "Missing expression to left of operator.";
    private static final String MISSING_RIGHT = "Missing expression to right of operator.";
    private static final String SYNTAX_ERROR = "Syntax error in expression.";
    private static final String PARSER_ERROR = "Parser error in expression.";

    /**
     * Builds an <tt>Expression</tt>.
     *
     * @param elements a <tt>List</tt> of <tt>ExpressionElements</tt>.
     * @return the complete <tt>Expression</tt>.
     */
   public static Expression build(List elements) throws TemplateException {
		groupParens(elements);
		return buildExpression(elements);
    }

    /**
     * Builds an <tt>Expression</tt> or sub-<tt>Expression</tt>.
     *
     * @param elements a <tt>List</tt> of <tt>ExpressionElements</tt>.
     * @return the complete <tt>Expression</tt> or
     * sub-<tt>Expression</tt>.
     */
    private static Expression buildExpression(List elements) throws TemplateException {
		// In descending order of operator precendence, associate each kind of operator
		// with its operands.
		int level = 0;
		while (elements.size() > 1 && level < numLevels) {
			associateOperators(elements, opClasses[level]);
			level++;
		}

		// The result should be a one-element list containing a
		// complete Expression.
		if (elements.size() == 1) {
			Object element = elements.get(0);
			
			// Remove extra parentheses.
			if (element instanceof List) {
				element = buildExpression((List)element);
			}
			
			if (element instanceof Expression) {
				Expression expression = (Expression)element;
				if (expression.isComplete()) {
					return expression;
				} else {
					// If the Expression isn't complete, it's
					// because the parser gave us an operator we
					// don't know about.
					throw new TemplateException(PARSER_ERROR);
				}
			} else {
				throw new TemplateException(SYNTAX_ERROR);
			}
		} else {
			throw new TemplateException(SYNTAX_ERROR);
		}
    }

    /**
     * Encapsulates parenthetical expressions by putting them in
     * sub-<tt>Lists</tt>.
     *
     * @param elements a <tt>List</tt> of <tt>ExpressionElements</tt>.
     */
    private static void groupParens(List elements) throws TemplateException {
		ListIterator iterator = elements.listIterator();
		Parenthesis paren;

		// Look for a paren.
		while ((paren = getNextParen(iterator)) != null) {

			// If the first one we find isn't an open paren, it's mismatched.
			if (paren instanceof CloseParen) {
				throw new TemplateException(MISMATCHED_PARENS);
			}

			// Eat up elements and put them in a sub-List, counting
			// parens until we get to the corresponding close paren.
			iterator.remove();
			LinkedList groupedElements = new LinkedList();
			int parenLevel = 1;

			while (iterator.hasNext()) {
				Object element = iterator.next();
				iterator.remove();
				if (element instanceof OpenParen) {
					parenLevel++;
				} else if (element instanceof CloseParen) {
					parenLevel--;
					if (parenLevel == 0) break;
				}
				groupedElements.add(element);
			}

			// If we didn't find it, there's a mismatch.
			if (parenLevel > 0) {
				throw new TemplateException(MISMATCHED_PARENS);
			}

			// If the parenthetical expression is empty, complain.
			if (groupedElements.isEmpty()) {
				throw new TemplateException(EMPTY_PARENS);
			}

			// Recurse to encapsulate any parenthetical expressions
			// in the sub-List.
			groupParens(groupedElements);

			// Insert it into the List.
			iterator.add(groupedElements);
		}
    }

    /**
     * Finds the next paren.
     *
     * @param iterator an <tt>Iterator</tt> for the
     * <tt>List</tt> of <tt>ExpressionElements</tt>.
     * @return the next paren found, or null if none found.
     */
    private static Parenthesis getNextParen(ListIterator iterator) {
		while (iterator.hasNext()) {
			Object element = iterator.next();
			if (element instanceof Parenthesis) {
				return (Parenthesis)element;
			}
		}
		return null;
    }

    /**
     * Associates operators with their operands.
     *
     * @param elements a <tt>List</tt> of <tt>ExpressionElements</tt>.
     * @param expressionClasses the subclasses of <tt>Expression</tt> that are at
	 * the current precedence level.
     */
    private static void associateOperators(List elements, Class [] expressionClasses)
		throws TemplateException {
		ListIterator iterator = elements.listIterator();

		// Loop through the elements.
		while (iterator.hasNext()) {
			Object element = iterator.next();

			// Build parenthetical subexpressions as we go along.
			if (element instanceof List) {
				iterator.set(buildExpression((List)element));

			} else {
				// If we find one of the operators we're looking
				// for, and it's incomplete, associate it with
				// its operand(s).
				Class elementClass = element.getClass();
				for (int i = 0; i < expressionClasses.length; i++) {
					if (elementClass == expressionClasses[i]) {
						if (element instanceof Binary) {
							Binary binary = (Binary)element;
							if (!binary.isComplete()) {
								// Skip back over the operator.
								iterator.previous();

								Expression left = getPreviousExpression(iterator);

								// Skip forward over the operator.
								iterator.next();

								Expression right = getNextExpression(iterator);

								binary.setLeft(left);
								binary.setRight(right);
								break;
							}
						} else if (element instanceof Unary) {
							Unary unary = (Unary)element;
							if (!unary.isComplete()) {
								// Associate the operator with the operand either
								// to the left or to the right, depending on the
								// operator's association type.
								Expression target;
								switch (unary.getAssociationType()) {
									case Unary.PREFIX:
										target = getNextExpression(iterator);
										break;
									case Unary.POSTFIX:
										// Skip back over the operator.
										iterator.previous();

										target = getPreviousExpression(iterator);

										// Skip forward over the operator.
										iterator.next();
										break;
									default:
										throw new TemplateException(PARSER_ERROR);
								}
								unary.setTarget(target);
							}
						} else {
							throw new TemplateException(PARSER_ERROR);
						}
					}
				}
			}
		}
    }

    /**
     * Gets the previous <tt>Expression</tt> in the list, and removes it.
     *
     * @param iterator an <tt>Iterator</tt> for the
     * <tt>List</tt> of <tt>ExpressionElements</tt>.
     * @return the previous <tt>Expression</tt>.
     */
    private static Expression getPreviousExpression(ListIterator iterator)
		throws TemplateException {
		if (!iterator.hasPrevious()) {
			throw new TemplateException(MISSING_LEFT);
		}
		Object element = iterator.previous();
		iterator.remove();
		if (!(element instanceof Expression)) {
			throw new TemplateException(MISSING_LEFT);
		} else {
			return (Expression)element;
		}
    }

    /**
     * Gets the next <tt>Expression</tt> in the list, builds it if
     * necessary, and removes it.
     *
     * @param iterator an <tt>Iterator</tt> for the
     * <tt>List</tt> of <tt>ExpressionElements</tt>.
     * @return the next <tt>Expression</tt>.
     */
    private static Expression getNextExpression(ListIterator iterator)
		throws TemplateException {
		if (!iterator.hasNext()) {
			throw new TemplateException(MISSING_RIGHT);
		}

		Object element = iterator.next();
		iterator.remove();

		if (element instanceof Expression) {
			return (Expression)element;
		} else if (element instanceof List) {
			return buildExpression((List)element);
		} else {
			throw new TemplateException(MISSING_RIGHT);
		}
    }
}
