/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.expression;

import freemarker.template.*;

/**
 * A unary operator that uses the string value of an expression as a hash key.
 * It associates with the <tt>Identifier</tt> or <tt>Dot</tt> to its left.
 */
public class DynamicKeyName extends Variable implements Unary {

    private Expression nameExpression;
    private Variable target;

    public DynamicKeyName() { }

    public DynamicKeyName(Expression nameExpression) {
        setNameExpression(nameExpression);
    }

    public void setNameExpression(Expression nameExpression) {
        this.nameExpression = nameExpression;
    }

    public void setTarget(Expression target) throws TemplateException {
        if (!(target instanceof Variable && target.isComplete())) {
            throw new TemplateException("Syntax error in expression.");
        }
        this.target = (Variable)target;
    }

    public Expression getTarget() {
        return target;
    }

    public boolean isComplete() {
        return !(nameExpression == null || target == null);
    }

    public int getAssociationType() {
        return POSTFIX;
    }

    public TemplateModel getAsTemplateModel(TemplateModelRoot modelRoot) throws TemplateException {
        TemplateModel targetModel = target.getAsTemplateModel(modelRoot);

        if (targetModel == null) return null;

        if(targetModel instanceof TemplateListModel) {
            int index;
            if((nameExpression instanceof NumberLiteral)) {
                index = ((NumberLiteral)nameExpression).intValue();
            } else {
                try {
                    index = Integer.parseInt(nameExpression.getValue(modelRoot));
                } catch(NumberFormatException ex) {
                    // Fall back to TemplateHashModel, if we can
                    if (targetModel instanceof TemplateHashModel) {
                        return getHashModel((TemplateHashModel)targetModel, modelRoot);
                    }
                    throw new TemplateException("Expression is not a valid index for " + target.getName(modelRoot));
                }
            }
            try {
                return ((TemplateListModel)targetModel).get(index);
            } catch(TemplateModelException ex) {
                // Fall back to TemplateHashModel, if we can
                if (targetModel instanceof TemplateHashModel) {
                    return getHashModel((TemplateHashModel)targetModel, modelRoot);
                }
                throw new TemplateException("IndexOutOfBoundsError in " + target.getName(modelRoot));
            }
        } 
        else if (targetModel instanceof TemplateHashModel) {

            return getHashModel((TemplateHashModel)targetModel, modelRoot);
        }
        else {
            throw new TemplateException(target.getName(modelRoot) +" is not a TemplateHashModel or a TemplateListModel or a TemplateHashModel, it's a " + targetModel.getClass().getName() + ".");
        }
    }

    public String getName(TemplateModelRoot modelRoot) throws TemplateException {
        return nameExpression.getValue(modelRoot);
    }

    /**
     * Gets the template model for a HashModel. Used when we know its a
     * TemplateHashModel we're looking at, and can't perform a lookup in
     * a TemplateListModel for some reason.
     */
    protected TemplateModel getHashModel(TemplateHashModel targetHash, 
            TemplateModelRoot modelRoot) throws TemplateException {

        TemplateModel referent = null;
        String keyName = nameExpression.getValue(modelRoot);

        try {
            referent = targetHash.get(keyName);
        } catch (TemplateModelException e) {
            throw new TemplateException("Couldn't get referent of " +
                                        keyName + ":\n" +
                                        Template.getStackTrace(e));
        }
        return referent;
    }
}