/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.expression;

import freemarker.template.*;

/**
 * The dot operator. Used to reference items inside a <code>TemplateHashModel</code>.
 */
public class Dot extends Variable implements Binary {
    protected Variable left;
    protected Identifier right;

    public String getName(TemplateModelRoot modelRoot) throws TemplateException {
        return right.getName(modelRoot);
    }

    public void setLeft(Expression left) throws TemplateException {
        if (!(left instanceof Variable)) {
            throw new TemplateException("Identifier required to left of dot.");
        }
        this.left = (Variable)left;
    }

    public void setRight(Expression right) throws TemplateException {
        if (!(right instanceof Identifier)) {
            throw new TemplateException("Identifier required to right of dot.");
        }
        this.right = (Identifier)right;
    }

    public Expression getLeft() {
        return left;
    }

    public Expression getRight() {
        return right;
    }

    public boolean isComplete() {
        return (left != null && right != null);
    }

    public TemplateModel getAsTemplateModel(TemplateModelRoot modelRoot) throws TemplateException {
        TemplateModel leftModel = left.getAsTemplateModel(modelRoot);

        if (leftModel == null) return null;

        TemplateModel referent = null;
        if (leftModel instanceof TemplateHashModel) {
            TemplateHashModel leftHash = (TemplateHashModel)leftModel;

            try {
                referent = leftHash.get(right.getName(modelRoot));
            } catch (TemplateModelException e) {
                throw new TemplateException("Couldn't get referent of " +
                                            right.getName(modelRoot) + ":\n" +
                                            Template.getStackTrace(e));
            }
        } else {

            throw new TemplateException(left.getName(modelRoot) +
                                        " is not a TemplateHashModel, it's a " +
                                        leftModel.getClass().getName() + ".");

        }

        return referent;
    }
}