/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.compiler;

import freemarker.template.*;
import freemarker.template.instruction.*;

/**
 * An abstract superclass for classes that parse templates.
 */
public abstract class TemplateParser {

    /**
     * The text to be parsed.
     */
    protected String text;
    protected int textLen;

	/**
	 * The Template being parsed.
	 */
	protected Template template;

    /**
     * The parse position.
     */
    protected int parsePos = 0;


    /**
     * The parse position before the last instruction was found.
     */
    protected int previousParsePos = 0;

    /**
     * The position at which the last instruction was found.
     */
    protected int foundPos = 0;

    public TemplateParser() { }

    /**
     * @param text the text to be parsed.
	 * @param template the Template being parsed.
     */
    public TemplateParser(Template template, String text) {
		setTemplate(template);
		setText(text);
    }

    /**
     * @param text the text to be parsed.
     */
    public void setText(String text) {
		this.text = text;
		textLen = text.length();
    }

	/**
	 * @param template the Template being parsed.
	 */
	public void setTemplate(Template template) {
		this.template = template;
	}

    /**
     * @param parsePos the parse position.
     */
    public void setParsePos(int parsePos) {
		this.parsePos = parsePos;
    }

    /**
     * @return the position at which the last instruction was found.
     */
    public int getFoundPos() {
		return foundPos;
    }

    /**
     * @return true if there is more text to parse.
     */
    public boolean hasMoreText() {
		return (parsePos < textLen);
    }

    /**
     * @return true if there is unparsed text before the last
     * instruction found.
     */
    public boolean hasTextBefore() {
		return (foundPos > previousParsePos);
    }

    /**
     * @return any text between the last parse position and the
     * last instruction found.
     */
    public String getTextBefore() {
		return text.substring(previousParsePos, foundPos);
    }

    /**
     * @return any remaining text.
     */
    public String getRemainingText() {
		return text.substring(parsePos);
    }

    /**
     * Searches the text for an instruction, starting at the current
     * parse position.  If one is found, parses it into an
     * <tt>Instruction</tt>.  Before changing <tt>parsePos</tt>, should set
     * <tt>previousParsePos = parsePos</tt>.  If no instruction is found,
     * should leave <tt>parsePos</tt> unchanged.
     *
     * @return a <tt>TemplateInstruction</tt> representing the
     * next instruction following the current parse position.
     */
    public abstract Instruction getNextInstruction() throws TemplateException;


    /**
     * Searches the text for a matching end instruction, starting at the
     * current parse position.  If we find it, parse it.  Before changing
     * <tt>parsePos</tt>, should set  <tt>previousParsePos = parsePos</tt>.
     * If no instruction is found, should leave <tt>parsePos</tt>
     * unchanged.
     *
     * @return true if we find the end instruction we're after, otherwise
     * false.
     */
     public abstract boolean skipToEndInstruction(ContainerInstruction beginInstruction);


    /**
     * Adds text to an error message indicatating the line
     * number where the error was found.
     *
     * @return a <tt>String</tt> containing the message.
     */
    public String atChar(int pos) {

          int lineNum = 1;
          for (int charParsed = 0; charParsed< pos ; charParsed++)
               if (text.charAt(charParsed)=='\n')
                    lineNum++;

//		return " at character " + String.valueOf(pos + 1);
          return " at line " + String.valueOf(lineNum);
    }
}
