/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.compiler;

import freemarker.template.*;
import freemarker.template.instruction.*;

/**
 * An abstract superclass for objects that build the compiled form of
 * a template.  Each instance can be used to compile one template.
 */
public abstract class TemplateBuilder {
	protected Template template;

	/**
	 * @param template the <tt>Template</tt> object that is to receive
	 * the compiled structure.
	 */
	public TemplateBuilder(Template template) {
		this.template = template;
	}

	/**
	 * @return a <tt>TemplateProcessor</tt> representing the compiled
	 * form of the template.
	 */
    public abstract TemplateProcessor build();
	
	/**
	 * When a subclass of <tt>TemplateBuilder</tt> calls
	 * <tt>Instruction.callBuilder()</tt>, the
	 * <tt>Instruction</tt> will call this method if it is an
	 * <tt>EmptyInstruction</tt>.
	 *
	 * @param instruction the <tt>Instruction</tt> on which
	 * <tt>callBuilder()</tt> was called.
	 * @param element the <tt>RuntimeStructuralElement</tt> that is to
	 * receive the <tt>Instruction</tt> as its statement.
	 */
	public abstract void buildStatement(EmptyInstruction instruction,
		RuntimeStructuralElement element) throws TemplateException;

	/**
	 * When a subclass of <tt>TemplateBuilder</tt> calls
	 * <tt>Instruction.callBuilder()</tt>, the
	 * <tt>Instruction</tt> will call this method if it should
	 * be built as a <tt>GenericStartInstruction</tt>.
	 *
	 * @param instruction the <tt>Instruction</tt> on which
	 * <tt>callBuilder()</tt> was called.
	 * @param element the <tt>RuntimeStructuralElement</tt> that is to
	 * receive the <tt>Instruction</tt> as its statement.
	 */
	public abstract void buildStatement(GenericStartInstruction instruction,
		RuntimeStructuralElement element) throws TemplateException;

	/**
	 * When a subclass of <tt>TemplateBuilder</tt> calls
	 * <tt>Instruction.callBuilder()</tt>, the
	 * <tt>Instruction</tt> will call this method if it is a
	 * <tt>FunctionInstruction</tt>.
	 *
	 * @param instruction the <tt>Instruction</tt> on which
	 * <tt>callBuilder()</tt> was called.
	 * @param element the <tt>RuntimeStructuralElement</tt> that is to
	 * receive the <tt>Instruction</tt> as its statement.
	 */
	public abstract void buildStatement(FunctionInstruction instruction,
		RuntimeStructuralElement element) throws TemplateException;
	
	/**
	 * When a subclass of <tt>TemplateBuilder</tt> calls
	 * <tt>Instruction.callBuilder()</tt>, the
	 * <tt>Instruction</tt> will call this method if it is an
	 * <tt>IfInstruction</tt>.
	 *
	 * @param instruction the <tt>Instruction</tt> on which
	 * <tt>callBuilder()</tt> was called.
	 * @param element the <tt>RuntimeStructuralElement</tt> that is to
	 * receive the <tt>Instruction</tt> as its statement.
	 */
	public abstract void buildStatement(IfInstruction instruction,
		RuntimeStructuralElement element) throws TemplateException;
	
	/**
	 * When a subclass of <tt>TemplateBuilder</tt> calls
	 * <tt>Instruction.callBuilder()</tt>, the
	 * <tt>Instruction</tt> will call this method if it is a
	 * <tt>SwitchInstruction</tt>.
	 *
	 * @param instruction the <tt>Instruction</tt> on which
	 * <tt>callBuilder()</tt> was called.
	 * @param element the <tt>RuntimeStructuralElement</tt> that is to
	 * receive the <tt>Instruction</tt> as its statement.
	 */
	public abstract void buildStatement(SwitchInstruction instruction,
		RuntimeStructuralElement element) throws TemplateException;


	/**
	 * When a subclass of <tt>TemplateBuilder</tt> calls
	 * <tt>Instruction.callBuilder()</tt>, the
	 * <tt>Instruction</tt> will call this method if it is an
	 * <tt>UnparsedInstruction</tt>.
	 *
	 * @param instruction the <tt>Instruction</tt> on which
	 * <tt>callBuilder()</tt> was called.
	 * @param element the <tt>RuntimeStructuralElement</tt> that is to
	 * receive the <tt>Instruction</tt> as its statement.
	 */
	public abstract void buildStatement(UnparsedInstruction instruction,
		RuntimeStructuralElement element) throws TemplateException;
}
