/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 *
 * 22 October 1999: Modified by Holger Arendt to parse method calls
 * in expressions.
 */

package freemarker.template.compiler;

import freemarker.template.*;
import freemarker.template.expression.*;
import freemarker.template.instruction.*;
import java.util.*;

/**
 * Parses standard template language and generates
 * <tt>Instructions</tt>.  Uses <tt>ExpressionBuilder</tt>
 * to build expressions.
 *
 * @version $Id: StandardTemplateParser.java,v 1.13 2001/06/19 11:41:18 run2000 Exp $
 */
public class StandardTemplateParser extends TemplateParser {

	/**
	 * Represents an unparsed tag.  Subclasses can call back
	 * the parser to have themselves parsed.
	 */
	private abstract class Tag {
		/**
		 * Returns a new parsed object, or calls a parser method
		 * that returns one, advancing parsePos.
		 */
		abstract Instruction parse() throws TemplateException;
	}

	// A Map of tag names to <tt>Tag</tt> objects, which are stored
	// as flywheels.
	private Map tagMap = new HashMap();

	/**
	 * Represents an unparsed long operator (an operator that's more than
	 * one character long).  Subclasses return new <tt>Expression</tt> objects.
	 */
	private abstract class LongOperator {
		/**
		 * Returns a new Expression object corresponding to the operator,
		 * and advances parsePos.
		 */
		abstract Expression parse() throws TemplateException;
	}

	// A Map of two-character operator strings to <tt>LongOperator</tt> objects,
	// which are stored as flywheels.
	private Map longOpMap = new HashMap();

	// True if the list index keyword terminated an expression.
	private boolean foundListIndexKeyword; 

        // True if the foreach index keyword terminated an expression.
        private boolean foundForeachIndexKeyword;

	private int parenLevel = 0;

    // Template syntax
    protected static final String VAR_INSTR_START_CHARS = "${";
    protected static final char VAR_INSTR_END_CHAR = '}';
    protected static final String LIST_TAG = "list";
    protected static final String LIST_INDEX_KEYWORD = "as";
    protected static final String LIST_END_TAG = "/list";
    protected static final String IF_TAG = "if";
    protected static final String ELSE_TAG = "else";
    protected static final String IF_END_TAG = "/if";
	protected static final String SWITCH_TAG = "switch";
	protected static final String SWITCH_END_TAG = "/switch";
	protected static final String CASE_TAG = "case";
	protected static final String BREAK_TAG = "break";
	protected static final String DEFAULT_TAG = "default";
	protected static final String ASSIGN_TAG = "assign";
	protected static final String INCLUDE_TAG = "include";
	protected static final String FUNCTION_TAG = "function";
	protected static final String FUNCTION_END_TAG = "/function";
	protected static final String COMPRESS_TAG = "compress";
	protected static final String COMPRESS_END_TAG = "/compress";
	protected static final String CALL_TAG = "call";
    protected static final char TAG_START_CHAR = '<';
    protected static final char TAG_END_CHAR = '>';
	protected static final char END_TAG_START_CHAR = '/';
	protected static final char QUOTE_CHAR = '\"';
	protected static final char ESCAPE_CHAR = '\\';
        protected static final String COMMENT_TAG = "comment";
        protected static final String COMMENT_END_TAG = "/comment";
    protected static final String NOPARSE_TAG = "noparse";
    protected static final String NOPARSE_TAG_END = "/noparse";
    protected static final String FOREACH_TAG = "foreach";
    protected static final String FOREACH_INDEX_KEYWORD = "in";
    protected static final String FOREACH_END_TAG = "/foreach";
    protected static final String TRANSFORM_TAG = "transform";
    protected static final String TRANSFORM_END_TAG = "/transform";

	// Length of operators that are more than one character long.
	protected static final int LONG_OPERATOR_LENGTH = 2;

    protected boolean lastWasIdentifier = false;

    public StandardTemplateParser() {
		super();
		init();
    }

    public StandardTemplateParser(Template template, String text) {
		super(template, text);
		init();
    }

	private void init() {
		// Map tag strings to Tag objects.
		tagMap.put(LIST_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return parseListStart();
			}
		});

		tagMap.put(LIST_END_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return new EndInstruction(Instruction.LIST_END);
			}
		});

		tagMap.put(IF_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return parseIfStart();
			}
		});

		tagMap.put(ELSE_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return new EndInstruction(Instruction.ELSE);
			}
		});

		tagMap.put(IF_END_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return new EndInstruction(Instruction.IF_END);
			}
		});

		tagMap.put(SWITCH_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return parseSwitch();
			}
		});

		tagMap.put(SWITCH_END_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return new EndInstruction(Instruction.SWITCH_END);
			}
		});

		tagMap.put(CASE_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return parseCase();
			}
		});

		tagMap.put(BREAK_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return new EndInstruction(Instruction.BREAK);
			}
		});

		tagMap.put(DEFAULT_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return parseDefault();
			}
		});

		tagMap.put(ASSIGN_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return parseAssign();
			}
		});

		tagMap.put(INCLUDE_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return parseInclude();
			}
		});

		tagMap.put(FUNCTION_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return parseFunction();
			}
		});

		tagMap.put(FUNCTION_END_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return new EndInstruction(Instruction.FUNCTION_END);
			}
		});

		tagMap.put(CALL_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return parseFunctionCall();
			}
		});

		tagMap.put(COMPRESS_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return new CompressInstruction();
			}
		});

		tagMap.put(COMPRESS_END_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return new EndInstruction(Instruction.COMPRESS_END);
			}
		});

		tagMap.put(COMMENT_TAG, new Tag() {
			Instruction parse() throws TemplateException {
                             return parseComment();
			}
		});

		tagMap.put(COMMENT_END_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return new EndInstruction(Instruction.COMMENT_END);
			}
		});

		tagMap.put(FOREACH_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return parseForeachStart();
			}
		});

		tagMap.put(FOREACH_END_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return new EndInstruction(Instruction.FOREACH_END);
			}
		});

		tagMap.put(NOPARSE_TAG, new Tag() {
			Instruction parse() throws TemplateException {
				return parseNoparseStart();
			}
		});

		tagMap.put(NOPARSE_TAG_END, new Tag() {
			Instruction parse() throws TemplateException {
				return new EndInstruction(Instruction.NOPARSE_END);
			}
		});

        tagMap.put(TRANSFORM_TAG, new Tag() {
            Instruction parse() throws TemplateException {
                return parseTransformStart();
            }
        });

        tagMap.put(TRANSFORM_END_TAG, new Tag() {
            Instruction parse() throws TemplateException {
                return new EndInstruction(Instruction.TRANSFORM_END);
            }
        });


		// Map long operator strings to LongOperator objects.
		longOpMap.put("==", new LongOperator() {
			Expression parse() throws TemplateException {
				parsePos += LONG_OPERATOR_LENGTH;
				return new Equals();
			}
		});

		longOpMap.put("!=", new LongOperator() {
			Expression parse() throws TemplateException {
				parsePos += LONG_OPERATOR_LENGTH;
				return new NotEquals();
			}
		});

		longOpMap.put("&&", new LongOperator() {
			Expression parse() throws TemplateException {
				parsePos += LONG_OPERATOR_LENGTH;
				return new And();
			}
		});

		longOpMap.put("||", new LongOperator() {
			Expression parse() throws TemplateException {
				parsePos += LONG_OPERATOR_LENGTH;
				return new Or();
			}
		});
	}

    /**
     * Searches the text for an instruction, starting at the current
     * parse position.  If one is found, parses it into a
     * <tt>Instruction</tt>.  Before changing <tt>parsePos</tt>, sets
     * <tt>previousParsePos = parsePos</tt>.  If no instruction is found,
     * leaves <tt>parsePos</tt> unchanged.
     *
     * @return a <tt>Instruction</tt> representing the
     * next instruction following parsePos, or null if none
     * is found.
     */
    public Instruction getNextInstruction() throws TemplateException {
		previousParsePos = parsePos;
		while (parsePos < textLen) {
			// If this is a variable instruction, parse it.
			if (text.startsWith(VAR_INSTR_START_CHARS, parsePos)) {
				foundPos = parsePos;
				return parseVariableInstruction();
			}

			// If this is an HTML-style tag, get its name.
			if (text.charAt(parsePos) == TAG_START_CHAR) {
				int tagStartPos = parsePos;
				parsePos++;
				if (findTagNameEnd()) {
					String tagName = text.substring(tagStartPos + 1, parsePos);

					// If we have a Tag object for this tag,
					// have the Tag object call us back to parse it.
					Tag tag = (Tag)tagMap.get(tagName);
					if (tag != null) {
						foundPos = tagStartPos;
						Instruction instruction = tag.parse();
						try {
							findTagEnd();
						} catch (TemplateException e) {
							String errorMessage = "Syntax error" + atChar(foundPos) +
								finishErrorMessage(e);
							throw new TemplateException(errorMessage);
						}
						return instruction;
					}
				} else {
					continue;
				}
			}
			parsePos++;
		}
		parsePos = previousParsePos;
		return null;
    }

    /**
     * Searches the text for a matching end instruction, starting at the
     * current parse position.  If we find it, parse it and return.  Before
     * changing <tt>parsePos</tt>, should set  <tt>previousParsePos = 
     * parsePos</tt>.  If no instruction is found, should leave 
     * <tt>parsePos</tt> unchanged.
     *
     * @return true if we find the end instruction we're after, otherwise
     * false.
     */
    public boolean skipToEndInstruction(ContainerInstruction beginInstruction) {
		previousParsePos = parsePos;

		while (parsePos < textLen) {

			// If this is an HTML-style tag, get its name.
			if (text.charAt(parsePos) == TAG_START_CHAR) {
				int tagStartPos = parsePos;
				parsePos++;

				if ((findTagNameEnd()) &&
						(text.charAt(tagStartPos + 1) == END_TAG_START_CHAR)) {
					String tagName = text.substring(tagStartPos + 1, parsePos);
					Tag tag = (Tag)tagMap.get(tagName);

					// If we have a Tag object for this tag, and it's an
					// end tag, have the Tag object call us back to parse it.
					if (tag != null) {
						foundPos = tagStartPos;
						try {
							Instruction endInstruction = tag.parse();
							if(beginInstruction.testEndInstruction(endInstruction)) {
								findTagEnd();
								return true;
							}
						} catch (TemplateException e) {
							// Obviously not the end tag we expected,
							// continue scanning!
						}
					}
				}
			} else {
				parsePos++;
			}
		}
		parsePos = previousParsePos;
		return false;
    }
    

	/**
	 * Advances parsePos through alphanumeric characters, to the position
	 * of the next whitespace character or <tt>TAG_END_CHAR</tt>; leaves parsePos
	 * unchanged if not found.
	 *
	 * @return true if found.
	 */
	protected boolean findTagNameEnd() {
		int tagNameEndPos = parsePos;
		while (tagNameEndPos < textLen) {
			char c = text.charAt(tagNameEndPos);
			if (Character.isWhitespace(c) || c == TAG_END_CHAR) {
				parsePos = tagNameEndPos;
				return true;
			} else if (isIdentifierChar(c) || c == END_TAG_START_CHAR) {
				tagNameEndPos++;
			} else {
				return false;
			}
		}
		return false;
	}

	/**
	 * Requires a <tt>TAG_END_CHAR</tt>, optionally preceded by whitespace, and advances
	 * parsePos after the <tt>TAG_END_CHAR</tt>.
	 */
	protected void findTagEnd() throws TemplateException {
		skipWhitespace();
		if (text.charAt(parsePos) != TAG_END_CHAR) {
			throw new TemplateException("Expected end of tag.");
		} else {
			parsePos++;
		}
	}

    /**
     * Parses and builds an <tt>Expression</tt>.
     *
	 * @param checkParens if true, parses only up to the first extra
	 * close paren.
     * @return the completed <tt>Expression</tt>.
     */
    protected Expression parseExpression(boolean checkParens) throws TemplateException {
		int startPos = parsePos;
		List tokens = tokenizeExpression(checkParens);
		try {
			return ExpressionBuilder.build(tokens);
		} catch (TemplateException e) {
			throw new TemplateException("Syntax error in expression" +
										atChar(startPos) + finishErrorMessage(e));
		}
    }

    /**
     * Parses an expression by tokenizing it into
     * <tt>ExpressionElement</tt>s.
     *
	 * @param checkParens if true, stops tokenizing if an extra close
	 * paren is found.
     * @return a <tt>List</tt> of <tt>ExpressionElement</tt>s.
     */
    protected List tokenizeExpression(boolean checkParens) throws TemplateException {
		parenLevel = 0;
		ExpressionElement element = parseElement(checkParens);
		if (element == null) {
			throw new TemplateException("Missing expression.");
		}

		LinkedList tokens = new LinkedList();
		tokens.add(element);

		while ((element = parseElement(checkParens)) != null) {
			tokens.add(element);
		}

		return tokens;
    }

    /**
     * @return the next <tt>ExpressionElement</tt> following
     * parsePos, or null if none is found.
	 *
	 * @param checkParens if true, returns null if an extra close paren
	 * is found.
     */
    protected ExpressionElement parseElement(boolean checkParens)
		throws TemplateException {

		skipWhitespace();
		char c = text.charAt(parsePos);

		// Look for an identifier.
		if (isIdentifierStartChar(c)) {
			String identifierString = parseIdentifierAsString();

			// Stop parsing the expression if we hit the list index keyword.
			if (identifierString.equals(LIST_INDEX_KEYWORD)  ) {
				foundListIndexKeyword = true;
				lastWasIdentifier = false;
				return null;
            } else if (identifierString.equals(FOREACH_INDEX_KEYWORD) ) {
 				foundForeachIndexKeyword = true;
				lastWasIdentifier = false;
				return null; 
			} else {
                lastWasIdentifier = true;
                return new Identifier(identifierString);
			}
		} else {
			// Look for a long operator.
			String possibleOpString = text.substring(parsePos, parsePos + LONG_OPERATOR_LENGTH);
			LongOperator longOp = (LongOperator)longOpMap.get(possibleOpString);

			// If we have a matching LongOperator object, ask it for a corresponding Expression
			// object, and return that object.
			if (longOp != null) {
				lastWasIdentifier = false;
				return longOp.parse();

			} else {
				// Look for the things we can identify by one character:
				// a single-character operator, a parenthesis, a bracket,
				// or a string literal.
				switch (c) {
				case '.':
					parsePos++;
					lastWasIdentifier = false;
					return new Dot();
				case '!':
					parsePos++;
					lastWasIdentifier = false;
					return new Not();
				case '+':
					parsePos++;
					lastWasIdentifier = false;
					return new Concatenate();
				case '(':
					parenLevel++;
					parsePos++;
					if (lastWasIdentifier == false) 
						return new OpenParen();
					else {
						lastWasIdentifier = false;
						return parseMethodCall();
					}
				case ')':
					parenLevel--;
					lastWasIdentifier = false;
					// If we were asked to check paren levels,
					// stop reading tokens if we get an extra
					// close paren; this could mean that we're at
					// the end of a function call.
					if (checkParens && parenLevel < 0) {
						return null;
					}
					parsePos++;
					return new CloseParen();
				case '[':
            		parsePos++;
                    if (lastWasIdentifier == false) 
						return parseListLiteral();
					else {
                        ExpressionElement element;
                        
                        // Hack to deal with syntaxes such as:
                        // foo["bar"]["baz"]
						lastWasIdentifier = false;
						element = parseDynamicKeyName();
                        lastWasIdentifier = true;
                        return element;
					}
					
                case '{':
                    lastWasIdentifier = false;
                    parsePos++;
                    return parseHashLiteral();
				case QUOTE_CHAR:
					lastWasIdentifier = false;
					return parseStringLiteral();
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
					return parseNumberLiteral();
				default:
					lastWasIdentifier = false;
					return null;
				}
			}
		}
    }

	/**
	 * Parses a <tt>DynamicKeyName</tt>.  Expects parsePos to be on the
	 * open bracket.
	 *
	 * @return a <tt>DynamicKeyName</tt>.
	 */
	protected DynamicKeyName parseDynamicKeyName() throws TemplateException {
		int startPos = parsePos;

		Expression nameExpression = parseExpression(false);
		if (text.charAt(parsePos) == ']') {
			parsePos++;
			return new DynamicKeyName(nameExpression);
		} else {
			throw new TemplateException("Missing closing delimiter for key expression, " +
										"or illegal character in expression," +
										atChar(startPos) + ".");
		}
	}

    /**
     * Parses a <tt>StringLiteral</tt>.  Expects parsePos to be on the
	 * open quote.
     *
     * @return a <tt>StringLiteral</tt>.
     */
    protected StringLiteral parseStringLiteral() throws TemplateException {
		int startPos = parsePos;
		parsePos++;
		boolean found = false;

		StringBuffer stringValueBuf = new StringBuffer();

	FIND_END:
		while (parsePos < textLen) {
			char currentChar = text.charAt(parsePos);
		TEST_CHAR:
			switch (currentChar) {
			case QUOTE_CHAR:
				parsePos++;
				found = true;
				break FIND_END;
			case ESCAPE_CHAR:
				parsePos++;
				if (parsePos < textLen) {
					stringValueBuf.append(text.charAt(parsePos));
					parsePos++;
					break TEST_CHAR;
				} else {
					break FIND_END;
				}
			default:
				stringValueBuf.append(currentChar);
				parsePos++;
				break TEST_CHAR;
			}
		}

		if (found) {
			return new StringLiteral(stringValueBuf.toString());
		} else {
			throw new TemplateException("Unterminated string literal" +
										atChar(startPos) + ".");
		}
    }

    protected ListLiteral parseListLiteral() throws TemplateException {
        int startPos = parsePos;
		List arguments = new LinkedList();

		try {
			while (true) {
				if (skipChar(']')) {
					break;
				}

				skipWhitespace();
				arguments.add(parseExpression(false));

				skipChar(',');
			}

		} catch (TemplateException e) {
			String errorMessage = "Syntax error in list literal" + atChar(startPos) +
				finishErrorMessage(e);
			throw new TemplateException(errorMessage);
		}

		return new ListLiteral(arguments);
    }

    protected HashLiteral parseHashLiteral() throws TemplateException {
        int startPos = parsePos;
		List arguments = new LinkedList();

		try {
			while (true) {
				if (skipChar('}')) {
					break;
				}

				skipWhitespace();
				arguments.add(parseExpression(false));

				skipChar(',');
			}

		} catch (TemplateException e) {
			String errorMessage = "Syntax error in hash literal" + atChar(startPos) +
				finishErrorMessage(e);
			throw new TemplateException(errorMessage);
		}

		return new HashLiteral(arguments);
    }

  protected NumberLiteral parseNumberLiteral() throws TemplateException {
               int startPos = parsePos;
               boolean found = false;
               StringBuffer stringValueBuf = new StringBuffer();
    FIND_END:
               while (parsePos < textLen) {
                       char currentChar = text.charAt(parsePos);
      TEST_CHAR:
                       switch (currentChar) {
        case ' ':
        case ']':      
        case ')':
        case '>':
          found = true;
          break FIND_END;
        default:
          stringValueBuf.append(currentChar);
          parsePos++;
          break TEST_CHAR;
                       }
               }
    
               if (found) {
                       return new NumberLiteral(stringValueBuf.toString());
               } else {
                       throw new TemplateException("Unterminated number literal " +
                                  atChar(startPos) + ".");
               }
       }


    /**
     * Parses a <tt>VariableInstruction</tt>.  Expects parsePos to be at the
	 * beginning of the VAR_INSTR_START_CHARS.
     *
     * @return a <tt>VariableInstruction</tt>.
     */
    protected VariableInstruction parseVariableInstruction() throws TemplateException {
		int startPos = parsePos;
		parsePos += VAR_INSTR_START_CHARS.length();

		Variable variable = parseVariable();
		if (text.charAt(parsePos) == VAR_INSTR_END_CHAR) {
			parsePos++;
			return new VariableInstruction(variable);
		} else {
			throw new TemplateException("Missing closing delimiter for variable, " +
										"or illegal character in variable," +
										atChar(startPos) + ".");
		}
	}

	/**
	 * Parses an <tt>Expression</tt> and ensures that it's a <tt>Variable</tt>.
	 *
     * @return a <tt>Variable</tt>.
     */
	protected Variable parseVariable() throws TemplateException {
		int startPos = parsePos;
		Expression variableExpression = parseExpression(false);
		if (variableExpression instanceof Variable) {
			return (Variable)variableExpression;
		} else {
			throw new TemplateException("Variable expected" + atChar(startPos) + ".");
		}
	}

	/**
	 * Parses an <tt>Identifier</tt>.  Expects whitespace to have been skipped.
	 *
	 * @return an <tt>Identifier</tt>.
     */
	protected Identifier parseIdentifier() throws TemplateException {
		return new Identifier(parseIdentifierAsString());
	}

	/**
	 * Tries to parse an identifier as a string.  Expects whitespace to have been
	 * skipped.
	 *
	 * @return a string containing the identifier.
	 */
	protected String parseIdentifierAsString() throws TemplateException {
		int startPos = parsePos;
		while (parsePos < textLen) {
			if (isIdentifierChar(text.charAt(parsePos))) {
				parsePos++;
			} else {
				if (parsePos > startPos && isIdentifierStartChar(text.charAt(startPos))) {
					return text.substring(startPos, parsePos);
				} else {
					throw new TemplateException("Identifier expected" + atChar(startPos) + ".");
				}
			}
		}
		throw new TemplateException("Unexpected end of file.");
    }

    /**
     * Parses a <tt>ListInstruction</tt>'s start tag.
     *
     * @return a <tt>ListInstruction</tt> initialized with the values
     * from the tag.
     */
    protected ListInstruction parseListStart() throws TemplateException {
		int startPos = parsePos;
		Variable listVariable;
		Identifier indexVariable;
		try {
			// Get the variable representing the object to be listed.
			listVariable = parseVariable();

			// Make sure the expression stopped at the list index keyword.
			if (foundListIndexKeyword) {
				foundListIndexKeyword = false;
			} else {
				throw new TemplateException("Expected '" + LIST_INDEX_KEYWORD + "'.");
			}

			// Get the index variable.
			skipWhitespace();
			indexVariable = parseIdentifier();
		} catch (TemplateException e) {
			String errorMessage = "Syntax error in list statement" + atChar(startPos)
				+ finishErrorMessage(e);
			throw new TemplateException(errorMessage);
		}
		return new ListInstruction(listVariable, indexVariable);

    }

    /**
     * Parses a <tt>ListInstruction</tt>'s start tag with the FOREACH keyword.
     *
     * @return a <tt>ListInstruction</tt> initialized with the values
     * from the tag.
     */
    protected ListInstruction parseForeachStart() throws TemplateException {
		int startPos = parsePos;
		Variable listVariable;
		Identifier indexVariable;
                Expression indexKeyword;
		try {
			// Get the index variable.
			skipWhitespace();
			indexVariable = parseIdentifier();

                        // search for the "in" keyword
                        parseElement(false);

			// Make sure the expression stopped at the foreach index keyword.
			if (foundForeachIndexKeyword) {
				foundForeachIndexKeyword = false;
			} else {
				throw new TemplateException("Expected '" + FOREACH_INDEX_KEYWORD + "'.");
			}

			// Get the variable representing the object to be listed.
			listVariable = parseVariable();

		} catch (TemplateException e) {
			String errorMessage = "Syntax error in list statement" + atChar(startPos)
				+ finishErrorMessage(e);
			throw new TemplateException(errorMessage);
		}
                // foreach and list are the same, so we just return a ListInstruction
		return new ListInstruction(listVariable, indexVariable);


    }


    /**
     * Parses an <tt>IfInstruction</tt>'s start tag.
     *
     * @return an <tt>IfInstruction</tt> initialized with the expression
     * in the tag.
     */
    protected IfInstruction parseIfStart() throws TemplateException {
		int startPos = parsePos;
		Expression condition;
		try {
			condition = parseExpression(false);
		} catch (TemplateException e) {
			String errorMessage = "Syntax error in if statement" + atChar(startPos)
				+ finishErrorMessage(e);
			throw new TemplateException(errorMessage);
		}
		return new IfInstruction(condition);
    }

    /**
     * Parses an <tt>AssignInstruction</tt>'s tag.
     *
     * @return an <tt>AssignInstruction</tt> initialized with the values
     * from the tag.
     */
    protected AssignInstruction parseAssign() throws TemplateException {
		int startPos = parsePos;
		Identifier variable;
		Expression value;
		try {
			// Get the variable to assign to.
			skipWhitespace();
			variable = parseIdentifier();

			skipWhitespace();
			// Skip an optional equals sign.
			if (text.charAt(parsePos) == '=') {
				parsePos++;
			}

			// Get the variable or literal to be assigned.
			value = parseExpression(false);
		} catch (TemplateException e) {
			String errorMessage = "Syntax error in assignment" + atChar(startPos) +
				finishErrorMessage(e);
			throw new TemplateException(errorMessage);
		}
		return new AssignInstruction(variable, value);
    }

    /**
     * Parses an <tt>IncludeInstruction</tt>'s tag.
     *
     * @return an <tt>IncludeInstruction</tt> initialized with the name
     * in the tag.
     */
    protected IncludeInstruction parseInclude() throws TemplateException {
		int startPos = parsePos;
		Expression templateName;
		try {
			templateName = parseExpression(false);
		} catch (TemplateException e) {
			String errorMessage = "Syntax error in include statement" + atChar(startPos) +
				finishErrorMessage(e);
			throw new TemplateException(errorMessage);
		}
		return new IncludeInstruction(template, templateName);
    }

    /**
     * Parses a <tt>SwitchInstruction</tt>'s tag.
     *
     * @return a <tt>SwitchInstruction</tt> initialized with the expression
     * in the tag.
     */
    protected SwitchInstruction parseSwitch() throws TemplateException {
		int startPos = parsePos;
		Variable testExpression;
		try {
			// Get the test variable.
			testExpression = parseVariable();
		} catch (TemplateException e) {
			String errorMessage = "Syntax error in switch statement" + atChar(startPos) +
				finishErrorMessage(e);
			throw new TemplateException(errorMessage);
		}
		return new SwitchInstruction(testExpression);
    }

    /**
     * Parses a <tt>CaseInstruction</tt>'s tag.
     *
     * @return a <tt>CaseInstruction</tt> initialized with the expression
     * in the tag.
     */
    protected CaseInstruction parseCase() throws TemplateException {
		int startPos = parsePos;
		Expression expression;
		try {
			// Get the expression.
			expression = parseExpression(false);
		} catch (TemplateException e) {
			String errorMessage = "Syntax error in case statement" + atChar(startPos) +
				finishErrorMessage(e);
			throw new TemplateException(errorMessage);
		}
		return new CaseInstruction(expression);
    }

    /**
     * Parses a default <tt>CaseInstruction</tt>'s tag.
     *
     * @return a <tt>CaseInstruction</tt> with default
	 * status set to true.
     */
    protected CaseInstruction parseDefault() throws TemplateException {
		CaseInstruction defaultCase = new CaseInstruction();
		defaultCase.setIsDefault(true);
		return defaultCase;
    }

	/**
	 * Parses a <tt>FunctionInstruction</tt>'s tag.
	 *
	 * @return a <tt>FunctionInstruction</tt> intialized with the
	 * argument names in the tag.
	 */
	protected FunctionInstruction parseFunction() throws TemplateException {
		int startPos = parsePos;
		String functionName;
		List argumentNames = new LinkedList();
		try {
			// Get the function's name.
			skipWhitespace();
			functionName = parseIdentifierAsString();

			// Parse argument names.
			requireChar('(');
			while (true) {
				if (skipChar(')')) {
					break;
				}

				skipWhitespace();
				argumentNames.add(parseIdentifierAsString());

				skipChar(',');
			}
		} catch (TemplateException e) {
			String errorMessage = "Syntax error in function declaration" + atChar(startPos) +
				finishErrorMessage(e);
			throw new TemplateException(errorMessage);
		}
		return new FunctionInstruction(functionName, argumentNames);
	}

    /**
     * Parses a <tt>MethodCall</tt>.
     *
     * @return a new <tt>MethodCall</tt> object initialized with the arguments.
     */
    protected MethodCall parseMethodCall() throws TemplateException {

		int startPos = parsePos;
		List arguments = new LinkedList();
		try {
			while (true) {
				if (skipChar(')')) {
					break;
				}

				skipWhitespace();
				arguments.add(parseExpression(true));

				skipChar(',');
			}

		} catch (TemplateException e) {
			String errorMessage = "Syntax error in MethodCall statement" + atChar(startPos) +
				finishErrorMessage(e);
			throw new TemplateException(errorMessage);
		}

		return new MethodCall(arguments);
    }

    /**
     * Parses a <tt>CallInstruction</tt>'s tag.
     *
     * @return a <tt>CallInstruction</tt> initialized with the arguments
     * in the tag.
     */
    protected CallInstruction parseFunctionCall() throws TemplateException {
		int startPos = parsePos;
		String functionName;
		List arguments = new LinkedList();
		try {
			// Get the function's name.
			skipWhitespace();
			functionName = parseIdentifierAsString();

			// Parse argument names.
			requireChar('(');
			while (true) {
				if (skipChar(')')) {
					break;
				}

				skipWhitespace();
				arguments.add(parseExpression(true));

				skipChar(',');
			}
		} catch (TemplateException e) {
			String errorMessage = "Syntax error in call statement" + atChar(startPos) +
				finishErrorMessage(e);
			throw new TemplateException(errorMessage);
		}
		return new CallInstruction(template, functionName, arguments);
	}


    /**
     * Parses a <tt>CommentInstruction</tt>'s tag.
     *
     * @return a <tt>CommentInstruction</tt>
     * 
     */
    protected CommentInstruction parseComment() throws TemplateException {
		int startPos = parsePos;
		Expression expression;
		try {
			// Get the single line comment.
                        // <comment "remark">...</comment>
			expression = parseExpression(false);
			return new CommentInstruction( expression );            
		} catch (TemplateException e) {
                        // without the single line comment
                        // <comment>...</comment>
			return new CommentInstruction();
		}
    }


    /**
     * Parses a <tt>NoParseInstruction</tt>'s tag.
     *
     * @return a <tt>NoParseInstruction</tt>
     * 
     */
	protected NoParseInstruction parseNoparseStart() throws TemplateException {
		return new NoParseInstruction();
	}


    /**
     * Parses a <tt>TransformInstruction</tt>'s tag. This tag consists of one
     * parameter: the variable representing the <code>TemplateTransformModel</code>
     * to be used for the transformation.
     *
     * @return a <tt>TransformInstruction</tt>
     * 
     */
    protected TransformInstruction parseTransformStart() throws TemplateException {
        Variable transformVariable;
        int startPos = parsePos;

        try {
            // Get the index variable.
            skipWhitespace();

            // Get the variable representing the object to be listed.
            transformVariable = parseVariable();

        } catch (TemplateException e) {
            String errorMessage = "Syntax error in transform statement" + atChar(startPos)
                + finishErrorMessage(e);
            throw new TemplateException(errorMessage);
        }

        return new TransformInstruction(transformVariable);
    }
 
 	/**
	 * Advances beyond any whitespace; then, if the next character matches
	 * a given character, advances beyond it and returns true, otherwise
	 * returns false.
	 */
	protected boolean skipChar(char c) throws TemplateException {
		skipWhitespace();
		if (text.charAt(parsePos) == c) {
			parsePos++;
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Requires a given character, optionally precededy by whitespace.
	 */
	protected void requireChar(char c) throws TemplateException {
		if (!skipChar(c)) {
			throw new TemplateException();
		}
	}

    /**
     * Advances parsePos beyond any whitespace.
     *
     * @throws a <tt>TemplateException</tt> if there are no more
     * characters after whitespace has been skipped.
     */
    protected void skipWhitespace() throws TemplateException {
		while (parsePos < textLen && Character.isWhitespace(text.charAt(parsePos))) {
			parsePos++;
		}
		if (parsePos < textLen) {
			return;
		} else {
			throw new TemplateException("Unexpected end of file.");
		}
    }

	/**
	 * Finishes an error message by adding an exception message, if there is one.
	 */
	protected String finishErrorMessage(Exception e) {
		if (e.getMessage() != null) {
			return ": " + e.getMessage();
		} else {
			return ".";
		}
	}

    /**
     * <p>Determines whether a character is legal in an identifier. An identifier
     * is either something like a tag name, such as "foreach", or a variable
     * name, such as myHash.</p>
     *
     * <p>In this implementation, an identifier can contain any letters, digits,
     * or underscore characters.</p>
     *
     * <p>Note that this method does not affect identifiers inside a
     * dynamic key name.</p>
     */
	protected boolean isIdentifierChar(char c) {
		return (Character.isLetterOrDigit(c) || c == '_');
	}

    /**
     * <p>Determines whether a character is legal at the start of an identifier. 
     * An identifier is either something like a tag name, such as "foreach", 
     * or a variable name, such as myHash.</p>
     *
     * <p>In this implementation, an identifier can contain any letters, digits,
     * or underscore characters.</p>
     *
     * <p>Note that this method does not affect identifiers inside a
     * dynamic key name.</p>
     */
	protected boolean isIdentifierStartChar(char c) {
		return (Character.isLetter(c));
	}
}
