/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template.compiler;

import freemarker.template.*;
import freemarker.template.instruction.*;

/**
 * Builds a template as a tree structure in which child nodes
 * are stored in <tt>TemplateLinkedList</tt>s.
 */
public class LinkedListTemplateBuilder extends TemplateBuilder {

    private TemplateLink head;
    private TemplateParser parser;

    public LinkedListTemplateBuilder(Template template) {
		super(template);
	}

    public LinkedListTemplateBuilder(Template template, TemplateParser parser) {
		super(template);
		setParser(parser);
    }

    public void setParser(TemplateParser parser) {
		this.parser = parser;
    }

    /**
     * Builds the template.
     *
     * @return the head of the built template.
     */
    public TemplateProcessor build() {
		TemplateLinkedList list = new TemplateLinkedList();
		try {
			Instruction endInstruction = buildLinks(list);
			if (endInstruction != null) {
				throw new TemplateException("Unexpected instruction" +
					parser.atChar(parser.getFoundPos()) + ".");
			}
		} catch (TemplateException e) {
			// If there's a compile-time error, the builder returns only
			// the error message.
			TextBlockInstruction errorBlock = new TextBlockInstruction();
			errorBlock.setText(Template.formatErrorMessage(e.getMessage()));
			return errorBlock;
		}
		return list;
    }

    /**
     * Builds a list of <tt>TemplateLink</tt>s.
     *
     * @param list the current <tt>TemplateLinkedList</tt> to which links are to be added.
     * @return an end instruction, if the current branch ends with one, otherwise null.
     */
    private Instruction buildLinks(TemplateLinkedList list)
		throws TemplateException {
		Instruction instruction = null;

		while (true) {
			TemplateLink currentLink = new TemplateLink();
			list.add(currentLink);

			// Search for an instruction.
			instruction = parser.getNextInstruction();

			// If there aren't any more instructions, put the rest of the file
			// in a TextBlockInstruction, use that as the statement for the current
			// link, and return.
			if (instruction == null) {
				TextBlockInstruction textBlock = new TextBlockInstruction();
				textBlock.setText(parser.getRemainingText());
				currentLink.setStatement(textBlock);
				return null;
			}

			// If there's text before the instruction, put it in a TextBlockInstruction,
			// and use that as the statement for the current link.
			if (parser.hasTextBefore()) {
				TextBlockInstruction textBlock = new TextBlockInstruction();
				textBlock.setText(parser.getTextBefore());
				currentLink.setStatement(textBlock);
			}

			// If we've been given an end instruction, return it.
			if (instruction.isEndInstruction()) {
				return instruction;
			}

			// If we just made a TextBlockInstruction to use as the current link's
			// statement, give the instruction to the next link, otherwise
			// use it for the current link.
			if (currentLink.getStatement() != null) {
				currentLink = new TemplateLink();
				list.add(currentLink);
			}

			// Have the instruction call us back to be built and inserted
			// into the link.
			instruction.callBuilder(this, currentLink);

			// If there's any text after the instruction, continue building
			// links.
			if (!parser.hasMoreText()) {
				return null;
			}
		}
    }

	public void buildStatement(EmptyInstruction instruction,
		RuntimeStructuralElement element) throws TemplateException {
		element.setStatement(instruction);
	}

    public void buildStatement(GenericStartInstruction instruction,
		RuntimeStructuralElement element) throws TemplateException {
		buildGenericBody(instruction);
		element.setStatement(instruction);
	}

	public void buildStatement(FunctionInstruction instruction,
		RuntimeStructuralElement element) throws TemplateException {
		element.setStatement(new NOOPInstruction());
		buildGenericBody(instruction);
		template.addFunction(instruction);
	}

    /**
     * Builds a GenericStartInstruction's body.
     */
    private void buildGenericBody(GenericStartInstruction instruction)
		throws TemplateException {
		int foundPos = parser.getFoundPos();
		TemplateLinkedList body = new TemplateLinkedList();
		Instruction endInstruction = null;
		endInstruction = buildLinks(body);
		instruction.setBody(body);

		// Make sure buildLinks() returned the right end instruction.
		if (endInstruction == null || !instruction.testEndInstruction(endInstruction)) {
			throw new TemplateException("Expected end instruction for " +
				instruction.getTypeName() + parser.atChar(foundPos) + ".");
		}
    }

    public void buildStatement(IfInstruction instruction,
		RuntimeStructuralElement element) throws TemplateException {
		int foundPos = parser.getFoundPos();

		// Recursively build the if block.
		TemplateLinkedList ifBlock = new TemplateLinkedList();
		Instruction nextInstruction = buildLinks(ifBlock);
		instruction.setIfBlock(ifBlock);

		if (nextInstruction == null) {
			throw new TemplateException("Expected end of if statement" +
				parser.atChar(foundPos) + ".");
		}

		// Recursively build the else block, if there is one.
		switch (nextInstruction.getEndType()) {
			case Instruction.IF_END:
				break;
			case Instruction.ELSE:
				TemplateLinkedList elseBlock = new TemplateLinkedList();

				// After an else, there should be an end if.
				Instruction endInstruction = buildLinks(elseBlock);

				if (endInstruction != null && endInstruction.getEndType() == Instruction.IF_END) {
					instruction.setElseBlock(elseBlock);
				} else {
					throw new TemplateException("Expected end of else block for if statement" +
						parser.atChar(foundPos) + ".");
				}
					
				break;
			default:
				throw new TemplateException("Expected end of if statement" +
					parser.atChar(foundPos) + ".");
		}

		element.setStatement(instruction);
    }

    public void buildStatement(SwitchInstruction instruction,
		RuntimeStructuralElement element) throws TemplateException {
		int foundPos = parser.getFoundPos();

		CaseInstruction caseInstruction = null;
		int lastType = Instruction.NONE;

		// Get blocks followed by end instructions, and check for end
		// instructions that are meaningful to us.
		while (true) {
			TemplateLinkedList body = new TemplateLinkedList();
			Instruction endInstruction = buildLinks(body);

			if (endInstruction == null) {
				throw new TemplateException("Expected end of switch structure" +
					parser.atChar(foundPos) + ".");
			}

			if (lastType == Instruction.CASE) {
				caseInstruction.setBody(body);
				instruction.addCase(caseInstruction);
			}

			int endType = endInstruction.getEndType();
			switch (endType) {
				case Instruction.CASE:
					caseInstruction = (CaseInstruction)endInstruction;
					break;
				case Instruction.BREAK:
					if (lastType == Instruction.CASE) {
						caseInstruction.setHasBreak(true);
					} else {
						throw new TemplateException("Unexpected break" +
							parser.atChar(parser.getFoundPos()) + ".");
					}
					break;
				case Instruction.SWITCH_END:
					element.setStatement(instruction);
					return;
				default:
					throw new TemplateException("Unexpected instruction" +
						parser.atChar(parser.getFoundPos()) + ".");
			}

			lastType = endType;
		}
	}

	/**
	 * When a subclass of <tt>TemplateBuilder</tt> calls
	 * <tt>Instruction.callBuilder()</tt>, the
	 * <tt>Instruction</tt> will call this method if it is an
	 * <tt>UnparsedInstruction</tt>.
	 *
	 * @param instruction the <tt>Instruction</tt> on which
	 * <tt>callBuilder()</tt> was called.
	 * @param element the <tt>RuntimeStructuralElement</tt> that is to
	 * receive the <tt>Instruction</tt> as its statement.
	 */
	public void buildStatement(UnparsedInstruction instruction,
			RuntimeStructuralElement element) throws TemplateException {
		int foundPos = parser.getFoundPos();
		boolean	bFound;

		bFound = parser.skipToEndInstruction( instruction );

        if( bFound ) {
			instruction.setText(parser.getTextBefore());
            element.setStatement( instruction );
		} else {
			throw new TemplateException("Expected end instruction for " +
				instruction.getTypeName() + parser.atChar(foundPos) + ".");
		}
	}
}
