/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template;

/**
 * A timer for objects that implement <tt>Updatable</tt>.
 * Creates a new thread, in which it periodically calls the
 * <tt>Updatable</tt>'s <tt>update()</tt> method.
 */
public class UpdateTimer implements Runnable {

    private Updatable target;
    private long delay;
    private Thread timerThread;
    private boolean keepTiming;
    
    public UpdateTimer() { }

    /**
     * Constructs the timer with the update target and update interval
     *
     * @param target the object to be updated.
     * @param delay the number of milliseconds between updates.
     */
    public UpdateTimer(Updatable target, long delay) {
		setTarget(target);
		setDelay(delay);
    }

    /**
     * Sets the target to be updated
     * 
     * @param target the object to be updated.
     */
    public void setTarget(Updatable target) {
		this.target = target;
    }
    
    /**
     * Sets the update interval
     *
     * @param delay the number of milliseconds between updates.
     */
    public void setDelay(long delay) {
		this.delay = delay;
    }

    /**
     * Begins periodic automatic updates of the target.
     * Since it can't be determined if an existing thread is 
     * Runnable or not, create and start a new thread
     */
    public void startTiming() {
		keepTiming = true;
		timerThread = new Thread(this);
        timerThread.start();
    }

    /**
     * Stops automatically updating the target.
     */
    public void stopTiming() {
		keepTiming = false;
        timerThread = null;
    }

    public void run() {
		while (keepTiming) {
			try {
				Thread.sleep(delay);
			} catch (InterruptedException e) {
                return;
            }
			if (keepTiming) {
				target.update();
			}
		}
    }
}
