/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template;

import freemarker.template.compiler.*;
import freemarker.template.instruction.*;
import java.io.*;
import java.util.*;

/**
 * <p>An application or servlet can instantiate a subclass of <tt>Template</tt>
 * to compile and process an HTML template.
 *
 * <p>You can pass the filename of the template to the constructor, in
 * which case it is compiled immediately.  Once compiled, the template is
 * stored in an an efficient data structure for later use.
 *
 * <p>To process the template and produce HTML, call the
 * <tt>process()</tt> method, which takes a tree of
 * <tt>TemplateModel</tt> objects as its data model.  The root
 * node of the tree must be a <tt>TemplateModelRoot</tt>.
 *
 * <p>Any error messages from exceptions thrown by the data model, or
 * generated by the <tt>Template</tt> during compilation or
 * processing, will be included as HTML comments in the output.
 *
 * <p>To facilitate multithreading, <tt>Template</tt> objects are
 * immutable; if you need to recompile a template, you must make a new
 * <tt>Template</tt> object.  In most cases, it will be sufficient
 * to let a <tt>TemplateCache</tt> do this for you.
 *
 * @see TemplateCache
 */
 
public class Template implements TemplateProcessor {
    protected static final String COMMENT_START = "<!-- ";

    protected static final String LINE_SEPARATOR = System.getProperty("line.separator");
    protected static final String COMMENT_END = " -->"  + LINE_SEPARATOR;

    protected TemplateProcessor compiledTemplate;
	protected TemplateCache cache;
	protected Map functions = new HashMap();

    /**
     * Constructs an empty template.
     */
    public Template() { }

    /**
     * Constructs a template by compiling it from a file.  Calls
     * <tt>compileFromFile()</tt>.
     *
     * @param filePath the absolute path of the template file
     * to be compiled.
     */
    public Template(String filePath) throws IOException {
		compileFromFile(filePath);
    }

    /**
     * Constructs a template by compiling it from a file.  Calls
     * <tt>compileFromFile()</tt>.
     *
     * @param file a <tt>File</tt> representing the
     * template file to be compiled.
     */
    public Template(File file) throws IOException {
		compileFromFile(file);
    }

    /**
     * Constructs a template by compiling it from an
     * <tt>InputStream</tt>.  Calls
     * <tt>compileFromStream()</tt>.
     *
     * @param stream an <tt>InputStream</tt> from which the
     * template can be read.
     */
    public Template(InputStream stream) throws IOException {
		compileFromStream(stream);
    }

    /**
     * Constructs a template by compiling it from an
     * <tt>Reader</tt>.  Calls <tt>compileFromStream()</tt>.
     *
     * @param stream a <tt>Reader</tt> from which the
     * template can be read.
     */
	public Template(Reader stream) throws IOException {
		compileFromStream(stream);
	}

    /**
     * Reads and compiles a template from a file, by getting the file's
     * <tt>FileInputStream</tt> and using it to call <tt>compileFromStream()</tt>, using
     * the platform's default character encoding.
     *
     * @param filePath the absolute path of the template file
     * to be compiled.
     */
    public synchronized void compileFromFile(String filePath) throws IOException {
		File file = new File(filePath);
		compileFromFile(file);
    }

    /**
     * Reads and compiles a template from a file, by getting the file's
     * <tt>FileInputStream</tt> and using it to call <tt>compileFromStream()</tt>, using
     * the platform's default character encoding.
     *
     * @param file a <tt>File</tt> representing the
     * template file to be compiled.
     */
    public synchronized void compileFromFile(File file) throws IOException {
        FileInputStream inputStream;

		if (!file.exists()) {
			throw new FileNotFoundException("Template file " +
				file.getName() +
				" not found.");
		}
		if (!file.canRead()) {
			throw new IOException("Can't read from template file " +
				file.getName() +
				".");
		}

        inputStream = new FileInputStream(file);
        compileFromStream(inputStream);
        inputStream.close();
    }

    /**
     * Compiles the template from an <tt>InputStream</tt>, using the platform's default
	 * character encoding.  If the template has already been compiled, this method does
	 * nothing.
     *
     * @param stream an <tt>InputStream</tt> from which the
     * template can be read.
     */
    public synchronized void compileFromStream(InputStream stream) throws IOException {
        InputStreamReader streamReader = new InputStreamReader(stream);
        compileFromStream(streamReader);
        streamReader.close();
	}


    /**
     * Compiles the template from an <tt>Reader</tt>.  If the template has
     * already been compiled, this method does nothing.
     *
     * @param stream an <tt>Reader</tt> from which the
     * template can be read.
     */
    public synchronized void compileFromStream(Reader stream) throws IOException {
		if (compiledTemplate != null) {
			return;
		}

		StringBuffer textBuf = new StringBuffer();
		BufferedReader br = new BufferedReader(stream);
		String line;

		try {
			while ((line = br.readLine()) != null) {
				textBuf.append(line).append(LINE_SEPARATOR);
			}
		} catch (IOException e) {
			try {
				br.close();
			} catch (IOException e1) { }
			throw e;
		}

		br.close();
		compiledTemplate = compileText(textBuf.toString());
    }

    /**
     * Compiles the template text using the standard parser and builder
     * classes.  To provide different compilation behavior, subclasses
	 * need only override this method.
     *
     * @param text the text to compile.
     * @return a <tt>TemplateProcessor</tt> representing the compiled template.
     */
    protected synchronized TemplateProcessor compileText(String text) {
		StandardTemplateParser parser = new StandardTemplateParser(this, text);
		LinkedListTemplateBuilder builder = new LinkedListTemplateBuilder(this, parser);
		return builder.build();
	}

	/**
	 * Sets the <tt>TemplateCache</tt> that this template belongs to.
	 * <tt>IncludeInstruction</tt> objects will be able to request this
	 * <tt>TemplateCache</tt> at run-time.
	 *
	 * @param cache the <tt>TemplateCache</tt> that this template belongs to.
	 */
	public void setTemplateCache(TemplateCache cache) {
		this.cache = cache;
	}

	/**
	 * Gets the <tt>TemplateCache</tt> that this template belongs to.
	 *
	 * @return the TemplateCache that this template belongs to.
	 */
	public TemplateCache getTemplateCache() {
		return cache;
	}

	/**
	 * Adds a function to the template.  Called by the <tt>TemplateBuilder</tt>
	 * at compile-time.
	 */
	public void addFunction(FunctionInstruction function) {
		functions.put(function.getName(), function);
	}

	/**
	 * Retrieves a function from the template.  Called by <tt>CallInstruction</tt>s
	 * at run-time.
	 */
	public FunctionInstruction getFunction(String name) {
		return (FunctionInstruction)functions.get(name);
	}

	/**
	 * @return a <tt>Set</tt> of function names (<tt>String</tt> objects)
	 * that have been defined for this template.
	 */
	public Set getFunctionNames() {
		return functions.keySet();
	}

    /**
     * Processes the template, using data from a template model, and outputs
     * the resulting HTML to a <tt>PrintWriter</tt>.
     *
     * @param modelRoot the root node of the data model.  If null, an
	 * empty data model is used.
     * @param out a <tt>PrintWriter</tt> to output the HTML to.
     */
    public void process(TemplateModelRoot modelRoot, PrintWriter out) {
		if (compiledTemplate != null) {
			if (modelRoot == null) {
				modelRoot = new SimpleHash();
			}
			compiledTemplate.process(modelRoot, out);
		}
    }

    /**
     * Processes the template, using an empty data model, and outputs
     * the resulting HTML to a <tt>PrintWriter</tt>.
     *
     * @param out a <tt>PrintWriter</tt> to output the HTML to.
     */
    public void process(PrintWriter out) {
		SimpleHash modelRoot = new SimpleHash();
		process(modelRoot, out);
    }

    /**
     * Used by FreeMarker classes to format an error message as an HTML comment.
     */
    public static String formatErrorMessage(String errorMessage) {
		return COMMENT_START + "Template Error: " + errorMessage + COMMENT_END;
    }

    /**
     * Used by FreeMarker classes to format a stack trace as a string.
     */
    public static String getStackTrace(Exception e) {
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);
		e.printStackTrace(pw);
		pw.close();
		return sw.toString();
    }
}
