/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template;

import java.io.*;

/**
 * A simple implementation of the <tt>TemplateScalarModel</tt>
 * interface, using a <tt>String</tt>.
 *
 * <p>All the public methods in this implementation are synchronized.
 */
public class SimpleScalar implements TemplateScalarModel, Serializable {

    /**
     * @serial the String that this SimpleScalar wraps.
     */
    protected String value;

    /**
     * @serial if this SimpleScalar wraps a boolean, this variable is used
     */
    protected boolean booleanValue;
     
    /**
     * @serial if this SimpleScalar wraps a boolean value, this is set to true
     */
    protected boolean useBoolean = false;

    /**
     * Constructs an empty <tt>SimpleScalar</tt>.
     */
    public SimpleScalar() { }

    /**
     * Constructs a <tt>SimpleScalar</tt> containing a string value.
     *
     * @param value the string value.
     */
    public SimpleScalar(String value) {
		setValue(value);
    }

    /**
     * Constructs a <tt>SimpleScalar</tt> containing a boolean value.
     *
     * @param value the boolean value.
     */
    public SimpleScalar(boolean value) {
		setValue(value);
    }

    public synchronized boolean isEmpty() throws TemplateModelException {
                if( useBoolean ) return !booleanValue;
		return (value == null || value.equals("") );
    }

    public synchronized String getAsString() throws TemplateModelException {
                if( useBoolean ) return new Boolean( booleanValue ).toString();
		return value;
    }

    /**
     * Sets the string value of this <tt>SimpleScalar</tt>.
     *
     * @param value the string value.
     */
    public synchronized void setValue(String value) {

		this.value = value;
    }

    /**
     * Sets the boolean value of this <tt>SimpleScalar</tt>.
     *
     * @param value the boolean value.
     */
    public synchronized void setValue(boolean value) {
                this.useBoolean = true;
                this.booleanValue = value;
    }
}
