/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template;

import java.util.*;
import java.io.*;

/**
 * A simple implementation of the <tt>TemplateListModel</tt>
 * interface, using a <tt>LinkedList</tt>.
 *
 * <p>A <tt>SimpleList</tt> can act as a cache for another
 * <tt>TemplateListModel</tt>, e.g. one that gets data from a
 * database.  When passed another <tt>TemplateListModel</tt> as an
 * argument to its constructor or to its <tt>copy</tt> method, the
 * <tt>SimpleList</tt> immediately copies all the elements and
 * discards the <tt>TemplateListModel</tt>.
 *
 * <p>A <tt>SimpleList</tt> can also be given a shelf life; it
 * then keeps track of the time elapsed since the last time it copied
 * another <tt>TemplateListModel</tt>, or the last time its shelf
 * life was set.  Its owner can check whether it has become stale,
 * and refresh it if necessary by copying data into it again.
 *
 * <p>All the public methods in this implementation are synchronized.
 */
public class SimpleList implements TemplateListModel, Serializable {

    private static final int MS_PER_MINUTE = 60000;

    /**
     * @serial The <tt>LinkedList</tt> that this <tt>SimpleList</tt> wraps.
     */
    private LinkedList list = new LinkedList();
    private transient ListIterator iterator;

    /**
     * @serial The creation time to be used for determining staleness.
     */
    private long timeCreated;

    /**
     * @serial The shelf life in milliseconds of this <tt>SimpleList</tt>.
     */
    private long shelfLife;

    /**
     * @serial Whether this <tt>SimpleList</tt> has an indefinite
     * shelf life.
     */
    private boolean isPermanent = true;

    /**
     * @serial A <tt>TemplateModelException</tt> thrown by a
     * <tt>TemplateListModel</tt> copied into this one.
     */
    private TemplateModelException copyException;

    /**
     * Constructs an empty <tt>SimpleList</tt> with an
     * indefinite shelf life.
     */
    public SimpleList() { }

    /**
     * Constructs a <tt>SimpleList</tt> from the given 
     * <code>LinkedList</code> with an
     * idenfinite shelf life.
     */
    public SimpleList( LinkedList list ) {
        this.list = list;
    }

     /** 
      * Constructs a <tt>SimpleList</tt> from the given
      * <code>Array</code> of <code>TemplateModels</code>
      */
     public SimpleList( TemplateModel[] list ) {
         for( int i=0; i<list.length; i++ ) {
             this.list.add( list[i] );
         }
     }

    /**
     * Constructs a <tt>SimpleList</tt>, copying into it the
     * values from another <tt>TemplateListModel</tt>.  If the
     * latter throws an exception, this <tt>SimpleList</tt> saves
     * the exception, and throws it when its own <tt>next()</tt>
     * method is called.
     *
     * @param listToCopy the list to be copied into this one.
     */
    public SimpleList(TemplateListModel listToCopy) {
        copy(listToCopy);
    }

    /**
     * Removes all the elements from this <tt>SimpleList</tt>.
     */
    public synchronized void clear() {
        iterator = null;
        list.clear();
    }

    /**
     * Sets the amount of time before this <tt>SimpleList</tt>
     * will become stale.
     *
     * @param minutes the number of minutes before this
     * <tt>SimpleList</tt> will become stale.
     */
    public synchronized void setShelfLife(int minutes) {
        isPermanent = false;
        shelfLife = minutes * MS_PER_MINUTE;
        setTimeCreated();
    }


    /**
     * @return this <tt>SimpleList</tt>'s shelf life in minutes.
     */
    public synchronized int getShelfLife() {
        return (int)(shelfLife / MS_PER_MINUTE);
    }

    /**
     * Gives this <tt>SimpleList</tt> an indefinite shelf life.
     */
    public synchronized void setPermanent() {
        isPermanent = true;
    }

    /**
     * @return true if this <tt>SimpleList</tt> has an
     * indefinite shelf life.
     */
    public synchronized boolean isPermanent() {
        return isPermanent;
    }

    /**
     * @return true if this <tt>SimpleList</tt> has become stale.
     */
    public synchronized boolean isStale() {
        return (!isPermanent && (new Date().getTime() - timeCreated >= shelfLife));
    }

    /**
     * Sets the time created to the current time.
     */
    private synchronized void setTimeCreated() {
        timeCreated = new Date().getTime();
    }

    /**
     * Discards the contents of this <tt>SimpleList</tt>, and
     * copies into it the values from another
     * <tt>TemplateListModel</tt>.  If the latter throws an
     * exception, this <tt>SimpleList</tt> saves the exception,
     * and throws it when its own <tt>next()</tt> method is
     * called.
     *
     * @param listToCopy the list to be copied into this one.
     */
    public synchronized void copy(TemplateListModel listToCopy) {
        setTimeCreated();
        clear();
        copyException = null;

        try {
            if (!listToCopy.isRewound()) {
                listToCopy.rewind();
            }

            while (listToCopy.hasNext()) {
                list.add(listToCopy.next());
            }
        } catch (TemplateModelException e) {
            copyException = e;
        }
    }


    public synchronized boolean isEmpty() throws TemplateModelException {
        return list.isEmpty();
    }

    public synchronized boolean isRewound() throws TemplateModelException {
        return (iterator == null);
    }

    /**
     * Adds a <tt>TemplateModel</tt> to the end of this
     * <tt>SimpleList</tt>.  Also moves the cursor to the
     * beginning of the list.
     *
     * @param element the <tt>TemplateModel</tt> to be added.
     */
    public synchronized void add(TemplateModel element) {
        iterator = null;
        list.add(element);
    }

    /**
     * Adds a string to the end of this <tt>SimpleList</tt>, by
     * first wrapping the string in a <tt>SimpleScalar</tt>.
     * Also moves the cursor to the beginning of the list.
     *
     * @param element the string to be added.
     */
    public synchronized void add(String s) {
        add(new SimpleScalar(s));
    }

    /**
     * Adds a boolean to the end of this <tt>SimpleList</tt>, by
     * first wrapping the boolean in a <tt>SimpleScalar</tt>.
     * Also moves the cursor to the beginning of the list.
     *
     * @param element the boolean to be added.
     */
    public synchronized void add(boolean b) {
        add(new SimpleScalar(b));
    }

    /**
     * Makes a list iterator if we don't have one already.
     */
    private synchronized void checkIterator() {
        if (iterator == null) {
            iterator = list.listIterator();
        }
    }

    public synchronized boolean hasNext() throws TemplateModelException {
        checkIterator();
        return iterator.hasNext();
    }

    public synchronized TemplateModel next() throws TemplateModelException {
        if (copyException != null) {
            throw copyException;
        }

        checkIterator();

        if (iterator.hasNext()) {
            return (TemplateModel)iterator.next();
        } else {
            throw new TemplateModelException("No more elements.");
        }
    }

    public synchronized void rewind() throws TemplateModelException {
        iterator = null;
    }


    public synchronized TemplateModel listSize() throws TemplateModelException {
        return new SimpleScalar("" + list.size());
    }

    public synchronized TemplateModel get(int i) throws TemplateModelException {
        if(i>=list.size()) throw new TemplateModelException("IndexOutOfBounds(" + i + ")");
        return (TemplateModel)list.get(i);
    }



}
