/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template;

import java.util.*;
import java.io.*;

/**
 * A simple implementation of the <tt>TemplateHashModel</tt> and
 * <tt>TemplateModelRoot</tt> interfaces, using a
 * <tt>HashMap</tt>.
 *
 * <p>All the public methods in this implementation are synchronized.
 */
public class SimpleHash implements TemplateModelRoot, Serializable {

    /**
     * @serial the HashMap that this SimpleHash wraps.
     */
    protected AbstractMap hash;

    /**
     * Constructs an empty <tt>SimpleHash</tt>.
     */
    public SimpleHash() {
      hash = new HashMap();
    }
  
  /**
   * Constructs a <tt>SimpleHash</tt> from the given <code>AbstractMap</code>
   *
   * @param hash The AbstractMap to use as the backing for this SimpleHash
   */
  public SimpleHash(AbstractMap hash) {
    this.hash = hash;
  }
  
    public synchronized boolean isEmpty() throws TemplateModelException {
		return hash.isEmpty();
    }

    /**
     * Puts a <tt>TemplateModel</tt> in the hash.
     *
     * @param key the name by which the <tt>TemplateModel</tt>
     * is identified in the template.
     * @param model the <tt>TemplateModel</tt> to store.
     */
    public synchronized void put(String key, TemplateModel model) {
		hash.put(key, model);
    }

    /**
     * Puts a string in the hash, by first wrapping the string in
	 * a <tt>SimpleScalar</tt>.
     *
     * @param key the name by which the resulting <tt>TemplateModel</tt>
     * is identified in the template.
     * @param s the string to store.
     */
    public synchronized void put(String key, String s) {
		hash.put(key, new SimpleScalar(s));
    }

    /**
     * Puts a boolean in the hash, by first wrapping the boolean in
	 * a <tt>SimpleScalar</tt>.
     *
     * @param key the name by which the resulting <tt>TemplateModel</tt>
     * is identified in the template.
     * @param b the boolean to store.
     */
    public synchronized void put(String key, boolean b) {
		hash.put(key, new SimpleScalar(b));
    }

    public synchronized TemplateModel get(String key) throws TemplateModelException {
		return (TemplateModel)hash.get(key);
    }

    public synchronized void remove(String key) {
		hash.remove(key);
    }

    public String toString() {
      return hash.toString();
    } 
}
