/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template;

import java.util.*;
import java.util.EventObject;
import java.util.EventListener;

/**
 * A generic event multicaster class.  The client owns an
 * object of this class, and calls the <tt>fireEvent()</tt> method
 * to fire events.
 */
public class GenericEventMulticaster {
    private LinkedList listenerList;

    public GenericEventMulticaster() {
		listenerList = new LinkedList();
    }

    public synchronized void addListener(EventListener listener) {
		listenerList.add(listener);
    }

    public synchronized void removeListener(EventListener listener) {
		ListIterator iterator = listenerList.listIterator();
		while (iterator.hasNext()) {
			if ((EventListener)iterator.next() == listener) {
				iterator.remove();
				break;
			}
		}
    }

	/**
	 * Fires an event to all the listeners of this multicaster,
	 * using a <tt>ListenerAdapter</tt>.
	 *
	 * @see ListenerAdapter
	 */
    public void fireEvent(EventObject event, ListenerAdapter adapter) {
		List listClone;

		// To avoid possible thread deadlocks that could occur if a
		// listener attempts to modify listenerList while we're sending
		// the event to that listener, clone the list, then send the
		// event from unsynchronized code.
		synchronized(this) {
			listClone = (List)listenerList.clone();
		}

		ListIterator iterator = listClone.listIterator();
		while (iterator.hasNext()) {
			EventListener listener = (EventListener)iterator.next();
			adapter.fireEvent(event, listener);
		}
    }
}
