/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template;

import java.util.*;
import java.util.EventObject;
import java.util.EventListener;
import java.io.*;

/**
 * A <tt>TemplateCache</tt> that loads templates from a filesystem.  Given a 
 * directory path, the cache assumes by default that all files in the directory 
 * are templates. It can optionally be given a filename suffix for templates.
 * 
 * <p>Before using a <tt>FileTemplateCache</tt>, you must call its 
 * <tt>startAutoUpdate()</tt> method to begin periodic updates.  The default 
 * loading policy is LOAD_ON_DEMAND: templates are loaded into the cache only 
 * when requested, each template's modification date is checked each time it is 
 * requested, and the periodic updates are used only to remove deleted templates
 * from the cache.  If the loading policy is set to PRELOAD, all templates are 
 * loaded when <tt>startAutoUpdate()</tt> is called, and all files are checked 
 * during each periodic update.  If template files will not be changed 
 * frequently, use PRELOAD with a long delay value for maximum 
 * performance. 
 *
 * <p>For maximum flexibility LOAD_AD_HOC mode exists so that all 
 * templates are loaded when <tt>startAutoUpdate</tt> is called but files are 
 * not refreshed periodically. Instead, one can write a client that will ask
 * the <tt>FileTemplateCache</tt> to update a single template via the
 * <tt>Updatable#update(String name)</tt> method. Applications with a large 
 * number of templates many of which are not frequently updated will work well 
 * with LOAD_AD_HOC mode. Since templates are updated 'ad hoc' in this mode
 * rather than periodically calling <tt>startAutoUpdate()</tt> is not
 * required.
 *
 * <p>The string argument to the <tt>getTemplate()</tt> method is interpreted 
 * as the template's path relative to the cache's root directory, using a 
 * forward slash (<tt>/</tt>) as a separator (this is to facilitate using URL 
 * path info to request templates).  For example, if a <tt>TemplateCache</tt> 
 * object was made for the directory <tt>templates</tt>, which contains a 
 * subdirectory <tt>foo</tt>, in which there is a template file called 
 * <tt>index.html</tt>, you would call <tt>getTemplate("foo/index.html")</tt> 
 * to retrieve that template.
 *
 * <p>The owner of the cache should implement the <tt>CacheListener</tt>
 * interface and register itself using <tt>addCacheListener()</tt>.
 *
 * <p>If the template cannot read its cache directory, the periodic updates 
 * will be cancelled until the next time <tt>startAutoUpdate()</tt> is called.
 *
 * @see TemplateCache
 * @see CacheListener
 * @see Updatable#update(String)
 */
public class FileTemplateCache implements TemplateCache, Updatable {

    /**
     * Used with <tt>setLoadingPolicy()</tt> to indicate that templates
     * should be loaded as they are requested.
     */
    public static final int LOAD_ON_DEMAND = 0;

    /**
     * Used with <tt>setLoadingPolicy()</tt> to indicate that templates
     * should be preloaded.
     */
    public static final int PRELOAD = 1;

    /**
     * Used with <tt>setLoadingPolicy()</tt> to indicate that templates
     * are preloaded but there is no automatic updating of them. Instead,
     * only named templates are updated when the cache is requested to do so.
     */
    public static final int LOAD_AD_HOC = 2;

    protected File cacheDir;
    private Map cache = new HashMap();
    private UpdateTimer timer;
    private long delay = 5000; // five seconds
    private int loadingPolicy = LOAD_ON_DEMAND;
    private GenericEventMulticaster multicaster = new GenericEventMulticaster();
    private String encoding;
    private String filenameSuffix;

    // A ListenerAdapter for cacheUnavailable events.
    private ListenerAdapter unavailableAdapter = new ListenerAdapter() {
        public void fireEvent(EventObject event, EventListener listener) {
            ((CacheListener)listener).cacheUnavailable((CacheEvent)event);
        }
    };

    // A ListenerAdapter for elementUpdated events.
    private ListenerAdapter updatedAdapter = new ListenerAdapter() {
        public void fireEvent(EventObject event, EventListener listener) {
            ((CacheListener)listener).elementUpdated((CacheEvent)event);
        }
    };

    // A ListenerAdapter for elementUpdateFailed events.
    private ListenerAdapter updateFailedAdapter = new ListenerAdapter() {
        public void fireEvent(EventObject event, EventListener listener) {
            ((CacheListener)listener).elementUpdateFailed((CacheEvent)event);
        }
    };

    // A ListenerAdapter for elementRemoved events.
    private ListenerAdapter removedAdapter = new ListenerAdapter() {
        public void fireEvent(EventObject event, EventListener listener) {
            ((CacheListener)listener).elementRemoved((CacheEvent)event);
        }
    };

    /**
     * Constructs an empty <tt>FileTemplateCache</tt>.
     */
    public FileTemplateCache() { }

    /**
     * Constructs a <tt>TemplateCache</tt> with a directory in which
     * it will look for template files.
     *
     * @param path the absolute path of the directory containing
     * templates for this cache.
     */
    public FileTemplateCache(String path) {
        setPath(path);
    }

    /**
     * Constructs a <tt>TemplateCache</tt> with a directory in which
     * it will look for template files.
     *
     * @param dir the directory containing templates for this cache.
     */
    public FileTemplateCache(File dir) {
        setDirectory(dir);
    }

    /**
     * Constructs a <tt>TemplateCache</tt> with a directory in which
     * it will look for template files, and a delay representing the
     * number of seconds between cache updates.
     *
     * @param path the absolute path of the directory containing
     * templates for this cache.
     * @param delay the number of seconds between cache updates.
     */
    public FileTemplateCache(String path, long delay) {
        this(path);
        setDelay(delay);
    }

    /**
     * Constructs a <tt>TemplateCache</tt> with a directory in which
     * it will look for template files, and a delay representing the
     * number of seconds between cache updates.
     *
     * @param dir the directory containing templates for this cache.
     * @param delay the number of seconds between cache updates.
     */
    public FileTemplateCache(File dir, long delay) {
        this(dir);
        setDelay(delay);
    }
    /**
     * Returns the loading policy currently in effect
     *
     * @return a loading policy value
     */
    public int getLoadingPolicy() {
        return loadingPolicy;
    }

    /**
     * Sets the loading policy for this <tt>FileTemplateCache</tt>.  If
     * <tt>LOAD_ON_DEMAND</tt>, templates will be loaded as they are 
     * requested, and each template's file modification date will be 
     * checked each time it is requested.  If <tt>PRELOAD</tt>, all 
     * templates in the cache directory and its subdirectories will
     * be loaded when the cache is started, and new templates will be 
     * added to the cache each time it is updated.  If <tt>LOAD_AD_HOC</tt>, 
     * all templates in the cache directory and its subdirectories will be 
     * loaded when the cache is created and a particular template file's 
     * modification date will be checked each time the client requests the 
     * update of that and only that template. 
     * Defaults to <tt>LOAD_ON_DEMAND</tt>.
     *
     * @param loadingPolicy cache mode
     */
    public void setLoadingPolicy(int loadingPolicy) {
        this.loadingPolicy = loadingPolicy;
        switch(loadingPolicy) {
            case LOAD_AD_HOC:
                stopAutoUpdate();
                loadTemplates();
                break;
            case PRELOAD: 
                startAutoUpdate();
                break;
            case LOAD_ON_DEMAND: 
                removeTemplates();
                startAutoUpdate();
                break;
        }
    }
    
    /**
     * Sets the template cache root directory
     *
     * @param path the absolute path of the directory containing
     * templates for this cache.
     */
    public void setPath(String path) {
        setDirectory(new File(path));
    }

    /**
     * Returns the template cache root directory
     *
     * @return the absolute path of the directory containing
     * templates for this cache.
     */
    public String getPath() {
        return cacheDir.toString();
    }

    /**
     * Sets the template cache root directory
     *
     * @param dir the root directory containing templates for this cache
     */
    public synchronized void setDirectory(File dir) {
        removeTemplates();
        this.cacheDir = dir;
    }

    /**
     * Returns the template cache root directory
     *
     * @return the root directory containing templates for this cache
     */
    public File getDirectory() {
        return cacheDir;
    }

    /**
     * Sets the interval between two cache updates. This is meaningful
     * only if the cache policy is set to LOAD_ON_DEMAND or PRELOAD.
     * Defaults to five seconds.
     *
     * @param delay the number of seconds between cache updates
     */
    public synchronized void setDelay(long delay) {
        this.delay = delay * 1000;
    }

    /**
     * Returns the interval between two cache updates. This is meaningful
     * only if the cache policy is set to LOAD_ON_DEMAND or PRELOAD.
     *
     * @return the number of seconds between cache updates
     */
    public long getDelay() {
        return delay / 1000;
    }

    /**
     * Sets the character encoding to be used when reading template files.
     *
     * @param encoding the name of the encoding to be used; this will be 
     * passed to the constructor of <tt>InputStreamReader</tt>.
     */
    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    /**
     * Returns the character encoding to be used when reading template files.
     *
     * @return the name of the encoding to be used; this will be 
     * passed to the constructor of <tt>InputStreamReader</tt>.
     */
    public String getEncoding() {
       return encoding;
    }


    /**
     * Sets the template suffix. If set, files that do not have this suffix 
     * will be ignored when read into the cache.
     *
     * @param filenameSuffix the optional filename suffix of template
     * files to be read for this cache.  
     */
    public void setFilenameSuffix(String filenameSuffix) {
        this.filenameSuffix = filenameSuffix;
    }

    /**
     * Returns the template suffix. If set, files that do not have this suffix 
     * will be ignored when read into the cache.
     *
     * @param the optional filename suffix of template
     * files to be read for this cache.  
     */
    public String getFilenameSuffix() {
        return filenameSuffix;
    }

    public void addCacheListener(CacheListener listener) {
        multicaster.addListener(listener);
    }

    public void removeCacheListener(CacheListener listener) {
        multicaster.removeListener(listener);
    }

    /**
     * Gets a template from the cache
     *
     * @param name the template's filename, including its
     * path relative to the cache's root directory.
     *
     * @return the corresponding <tt>Template</tt>, or null
     * if not found or an error has occurred
     */
    public synchronized Template getTemplate(String name) {
        if (!(filenameSuffix == null || name.endsWith(filenameSuffix))) {
            fireCacheEvent(updateFailedAdapter, name,
                    new IOException("The requested name, \"" + name +
                    "\", does not have the filename suffix \"" +
                    filenameSuffix + "\"."));
            return null;
        }
                
    /* 
     * If we're preloading, then we can just return the template from the cache
     * if we have it -- it will be updated the next time our update() method is
     * called. Similarly, if we're loading on an ad hoc basis the template will 
     * be updated the next time our update(String name) method is called. On 
     * the other hand if we're loading on the demand, we have to check if 
     * the template has been changed since we last  cached it.
     */
        switch (loadingPolicy) {
            case LOAD_AD_HOC:
            case PRELOAD: 
                CacheElement element = (CacheElement)cache.get(name);
                return element != null ? (Template)element.object : null;
            case LOAD_ON_DEMAND: // If we're loading on demand, check the file.
                if (checkCacheDir())
                    return doUpdate(name);
            default:
                return null;
        } 
    }

    public void startAutoUpdate() {
        if(loadingPolicy != LOAD_AD_HOC) {
            synchronized (this) {
                stopAutoUpdate();
                if (timer == null) {
                    timer = new UpdateTimer(this, delay);
                }
                update();
                timer.startTiming();
            }
        }
    }

    public void stopAutoUpdate() {
        if (timer != null) {
            timer.stopTiming();
            timer = null;
        }
    }

    /**
     * Returns a list of cached files
     * @return a list of cached files
     */
    public Iterator listCachedFiles() {
        return Collections.unmodifiableCollection(cache.values()).iterator();
    }

    /**
     * Tells whether the template cache root directory exists and is readable
     *
     * @return true if the cache directory exists and is readable;
     * otherwise, also fires an appropriate <tt>CacheEvent</tt>.
     */
    private boolean checkCacheDir() {
        if (!cacheDir.exists()) {
            fireCacheEvent(unavailableAdapter, null, 
                    new IOException("Directory \"" +
                    cacheDir.getAbsolutePath() + "\" not found."));
            stopAutoUpdate();
            return false;
        }
        if (!cacheDir.isDirectory()) {
            fireCacheEvent(unavailableAdapter, null,
                   new IOException("\"" + cacheDir.getAbsolutePath() +
                   "\" is not a directory."));
            stopAutoUpdate();
            return false;
        }
        return true;
    }

    /**
     * Loads the template from a file
     *
     * @param file template file
     * @return template instance
     */
    private Template loadTemplate(File file) throws IOException {
        if (encoding == null) {
            return new Template(file);
        } else {
            FileInputStream inputStream = new FileInputStream(file);
            InputStreamReader streamReader = new InputStreamReader(inputStream, encoding);
            Template template = new Template(streamReader);
            streamReader.close();
            inputStream.close();
            return template;
        }
    }

    /**
     * Given a file, returns the template from the file if it's more
     * recent then the one in the cache; otherwise, returns the one
     * in the cache.
     *
     * @param elementName the name the template should have in the cache.
     * @param file the file from which the template may be read.
     */
    private Template getLatestTemplate(String elementName, File file) {

        Template template = null;
        long lastModified = file.lastModified();
        CacheElement element = (CacheElement)cache.get(elementName);

        try {
            // If we don't have this template, add it.
            if (element == null) {
                template = loadTemplate(file);
                template.setTemplateCache(this);
                cache.put(elementName, 
                        new CacheElement(elementName, template, lastModified));
                fireCacheEvent(updatedAdapter, elementName, null);

            // If we have it and it's been modified, update it.
            } else if (lastModified > element.lastModified) {
                template = loadTemplate(file);
                template.setTemplateCache(this);
                element.object = template;
                element.lastModified = lastModified;
                fireCacheEvent(updatedAdapter, elementName, null);
            } else {
            // Return the Template object we already have.
                template = (Template)element.object;
            }
        } catch (IOException e) {
            fireCacheEvent(updateFailedAdapter, elementName, e);
        }
        return template;
    }

    /**
     * Update a named template if in the LOAD_AD_HOC mode
     * Do nothing if in other modes
     * @param name of template to update
     */
	public synchronized void update(String name) {
		if (loadingPolicy == FileTemplateCache.LOAD_AD_HOC) {
			if (checkCacheDir()) 
                doUpdate(name);
        }
	}

    /**
     * Translates the name of the template to its file representation,
     * checks for the latter's existence and makes sure it's not a directory
     *
     * @param name template name
     * @return updated template
     */
	private Template doUpdate(String name) {

		File file = nameToFile(name);

		if (!file.exists()) {
            removeTemplate(name);
			fireCacheEvent(updateFailedAdapter, name, 
				new IOException("\""+file.getPath()+"\" doesn't exist"));
				return null;
		}
			
		if (!file.isFile()) {
			fireCacheEvent(updateFailedAdapter, name, 
				new IOException("\""+file.getAbsolutePath()+
                    "\" is a directory"));
			return null;
		}
			
		return getLatestTemplate(name, file);
	}

    /**
     * Updates the cache. In LOAD_AD_HOC mode, this does nothing.
     */
    public synchronized void update() {
        if (!checkCacheDir()) {
            return;
        }
        switch (loadingPolicy) {
            case LOAD_AD_HOC:
                // Do nothing, ad hoc is not auto updated
                break;
            case LOAD_ON_DEMAND:
                removeDeletedTemplates();
                break;
            case PRELOAD:
                loadTemplates();
                break;
            default:
                break;
        }
    }

    /**
     * Load all templates and remove the ones in the cache
     */
    private void loadTemplates() {
        LinkedList visitedFiles = new LinkedList();
        try {
            readDirectory(cacheDir, "", visitedFiles);
            removeUnvisitedTemplates(visitedFiles);
        } catch (IOException e) {
            stopAutoUpdate();
            fireCacheEvent(unavailableAdapter, null, e);
            return;
        }
    }

    /**
     * Remove all templates from the cache
     */
    private void removeTemplates() {
        cache.clear();
    }

    /**
     * Remove from the cache any templates corresponding to files we
     * didn't just visit.
     * @param list of visited templates
     */
    private void removeUnvisitedTemplates(LinkedList visitedFiles) {

        Set keySet = cache.keySet();
        Iterator keyIterator = keySet.iterator();

        while (keyIterator.hasNext()) {
            String elementName = (String)keyIterator.next();
            if (!visitedFiles.contains(elementName)) {
                keyIterator.remove();
                fireCacheEvent(removedAdapter, elementName, null);
            }
        } 
    }
    /**
     * Removes from the cache templates that correspond to deleted files.
     */
    private void removeDeletedTemplates() {

        Set keySet = cache.keySet();
        Iterator keyIterator = keySet.iterator();

        while (keyIterator.hasNext()) {
           String elementName = (String)keyIterator.next();
           File file = nameToFile(elementName);
           if (!file.exists()) {
                keyIterator.remove();
                fireCacheEvent(removedAdapter, elementName, null);
           }
        }
    }                

    /**
     * Removes named template from the cache.
     * @param name of template
     */
    private void removeTemplate(String name) {
        cache.remove(name);
        fireCacheEvent(removedAdapter, name, null);
    } 

    /**
     * Recursively updates the cache from the files in a (sub)directory
     * and its subdirectories.
     *
     * @param dir the directory to be read.
     * @param relativeDirPath a string representing the directory's path
     * relative to the root cache directory.
     * @param visitedFiles a List of files that have been visited so far.
     */
    private void readDirectory(File dir, String relativeDirPath, 
            List visitedFiles) throws IOException {
        String[] filenames = dir.list();
        if (filenames == null) {
            throw new IOException("Could not get file list from directory \"" 
                + dir.getAbsolutePath() + "\".");
        }
        // Iterate through the items in the directory.
        for (int fileNum = 0; fileNum < filenames.length; fileNum++) {
            String filename = filenames[fileNum];
            File file = new File(dir, filename);
            String elementName = relativeDirPath + filename;
            // If the item is a file, see if we need to to read it.
            if (file.isFile()) {
            // If we have no filename suffix, or if we have one and this
            // file ends with it, check the file.
                if (filenameSuffix == null || 
                        filename.endsWith(filenameSuffix)) {
                    visitedFiles.add(elementName);
                    getLatestTemplate(elementName, file);
                }
            } else if (file.isDirectory()) {
            // If the item is a directory, recursively read it.
                readDirectory(file, elementName + "/", visitedFiles);
            }
        } 
    }

    /**
     * Converts a cache element name to a <tt>File</tt>.
     */
    private File nameToFile(String name) {
        // Replace forward slashes with the operating system's
        // file separator, if it's not a forward slash.
        if (File.separatorChar == '/') {
            return new File(cacheDir, name);
        } else {
            StringBuffer pathBuf = new StringBuffer();
            for (int i = 0; i < name.length(); i++) {
                char c = name.charAt(i);
                if (c == '/') {
                    pathBuf.append(File.separatorChar);
                } else {
                    pathBuf.append(c);
                }
            }
            return new File(cacheDir, pathBuf.toString());
        }
    }

    /**
     * A convenience method for firing a CacheEvent.
     *
     * @param adapter a <tt>ListenerAdapter</tt>.
     * @param elementName the name of the cache element in question, or null.
     * @param e an <tt>Exception</tt> to be included with the event, or null.
     */
    private void fireCacheEvent(ListenerAdapter adapter, String elementName, 
            Exception e) {
        CacheEvent event = new CacheEvent(this);
        event.setElementName(elementName);
        event.setException(e);
        multicaster.fireEvent(event, adapter);
    }
}
