/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template;

import java.util.*;
import java.util.EventObject;
import java.util.EventListener;
import java.io.*;

/**
 * A <tt>BinaryCache</tt> that loads binary files from a filesystem.  Given a 
 * directory path, the cache assumes by default that all files in the directory 
 * are binary. It can optionally be given a filename suffix to select specific
 * binary file types such as images.
 * 
 * <p>Before using a <tt>FileBinaryCache</tt>, you must call its 
 * <tt>startAutoUpdate()</tt> method to begin periodic updates.  The default 
 * loading policy is LOAD_ON_DEMAND: files are loaded into the cache only 
 * when requested, each file's modification date is checked each time it is 
 * requested, and the periodic updates are used only to remove deleted files
 * from the cache.  If the loading policy is set to PRELOAD, all files are 
 * loaded when <tt>startAutoUpdate()</tt> is called, and all files are checked 
 * during each periodic update.  If binary files will not be changed 
 * frequently, use PRELOAD with a long delay value for maximum 
 * performance. 
 *
 * <p>For maximum flexibility LOAD_AD_HOC mode exists so that all 
 * files are loaded when <tt>startAutoUpdate</tt> is called but they are 
 * not refreshed periodically. Instead, one can write a client that will ask
 * the <tt>FileBinaryCache</tt> to update a single binary object via the
 * <tt>Updatable#update(String name)</tt> method. Applications with a large 
 * number of files in cache many of which are not frequently updated will 
 * work well with LOAD_AD_HOC mode. Since files are updated 'ad hoc' in this
 * mode rather than periodically calling <tt>startAutoUpdate()</tt> is not
 * required.
 *
 * <p>The string argument to the <tt>getBinaryData()</tt> method is interpreted 
 * as the file's path relative to the cache's root directory, using a 
 * forward slash (<tt>/</tt>) as a separator (this is to facilitate using URL 
 * path info to request file).  For example, if a <tt>BinaryCache</tt> 
 * object was made for the directory <tt>images</tt>, which contains a 
 * subdirectory <tt>foo</tt>, in which there is an image called 
 * <tt>site.jpg</tt>, you would call <tt>getBinaryData("foo/site.jpg")</tt> 
 * to retrieve that file.
 *
 * <p>The owner of the cache should implement the <tt>CacheListener</tt>
 * interface and register itself using <tt>addCacheListener()</tt>.
 *
 * <p>If the file cannot be read from its directory, the periodic updates 
 * will be cancelled until the next time <tt>startAutoUpdate()</tt> is called.
 *
 * @see BinaryCache
 * @see CacheListener
 * @see Updatable#update(String)
 */
public class FileBinaryCache implements BinaryCache, Updatable {

    /**
     * Used with <tt>setLoadingPolicy()</tt> to indicate that files
     * should be loaded as they are requested.
     */
    public static final int LOAD_ON_DEMAND = 0;

    /**
     * Used with <tt>setLoadingPolicy()</tt> to indicate that files
     * should be preloaded.
     */
    public static final int PRELOAD = 1;

    /**
     * Used with <tt>setLoadingPolicy()</tt> to indicate that files
     * are preloaded but there is no automatic updating of them. Instead,
     * only named files are updated when the cache is requested to do so.
     */
    public static final int LOAD_AD_HOC = 2;

    private File cacheDir;
    private Map cache = new HashMap();
    private UpdateTimer timer;
    private long delay = 5000; // five seconds
    private int loadingPolicy = LOAD_ON_DEMAND;
    private GenericEventMulticaster multicaster = new GenericEventMulticaster();
    private String filenameSuffix;

    // A ListenerAdapter for cacheUnavailable events.
    private ListenerAdapter unavailableAdapter = new ListenerAdapter() {
        public void fireEvent(EventObject event, EventListener listener) {
            ((CacheListener)listener).cacheUnavailable((CacheEvent)event);
        }
    };

    // A ListenerAdapter for elementUpdated events.
    private ListenerAdapter updatedAdapter = new ListenerAdapter() {
        public void fireEvent(EventObject event, EventListener listener) {
            ((CacheListener)listener).elementUpdated((CacheEvent)event);
        }
    };

    // A ListenerAdapter for elementUpdateFailed events.
    private ListenerAdapter updateFailedAdapter = new ListenerAdapter() {
        public void fireEvent(EventObject event, EventListener listener) {
            ((CacheListener)listener).elementUpdateFailed((CacheEvent)event);
        }
    };

    // A ListenerAdapter for elementRemoved events.
    private ListenerAdapter removedAdapter = new ListenerAdapter() {
        public void fireEvent(EventObject event, EventListener listener) {
            ((CacheListener)listener).elementRemoved((CacheEvent)event);
        }
    };

    /**
     * Constructs an empty <tt>FileBinaryCache</tt>.
     */
    public FileBinaryCache() { }

    /**
     * Constructs a <tt>BinaryCache</tt> with a directory in which
     * it will look for files.
     *
     * @param path the absolute path of the directory containing
     * files for this cache.
     */
    public FileBinaryCache(String path) {
        setPath(path);
    }

    /**
     * Constructs a <tt>BinaryCache</tt> with a directory in which
     * it will look for files.
     *
     * @param dir the directory containing files for this cache.
     */
    public FileBinaryCache(File dir) {
        setDirectory(dir);
    }

    /**
     * Constructs a <tt>BinaryCache</tt> with a directory in which
     * it will look for files, and a delay representing the
     * number of seconds between cache updates.
     *
     * @param path the absolute path of the directory containing
     * files for this cache.
     * @param delay the number of seconds between cache updates.
     */
    public FileBinaryCache(String path, long delay) {
        this(path);
        setDelay(delay);
    }

    /**
     * Constructs a <tt>BinaryCache</tt> with a directory in which
     * it will look for files, and a delay representing the
     * number of seconds between cache updates.
     *
     * @param dir the directory containing files for this cache.
     * @param delay the number of seconds between cache updates.
     */
    public FileBinaryCache(File dir, long delay) {
        this(dir);
        setDelay(delay);
    }
    /**
     * Returns the loading policy currently in effect
     *
     * @return a loading policy value
     */
    public int getLoadingPolicy() {
        return loadingPolicy;
    }

    /**
     * Sets the loading policy for this <tt>FileBinaryCache</tt>.  If
     * <tt>LOAD_ON_DEMAND</tt>, files will be loaded as they are 
     * requested, and each file's modification date will be 
     * checked each time it is requested.  If <tt>PRELOAD</tt>, all 
     * files in the cache directory and its subdirectories will
     * be loaded when the cache is started, and new files will be 
     * added to the cache each time it is updated.  If <tt>LOAD_AD_HOC</tt>, 
     * all files in the cache directory and its subdirectories will be 
     * loaded when the cache is created and a particular file's 
     * modification date will be checked each time the client requests the 
     * update of that and only that file. Defaults to <tt>LOAD_ON_DEMAND</tt>.
     *
     * @param loadingPolicy cache mode
     */
    public void setLoadingPolicy(int loadingPolicy) {
        this.loadingPolicy = loadingPolicy;
        switch(loadingPolicy) {
            case LOAD_AD_HOC:
                stopAutoUpdate();
                loadBinaries();
                break;
            case PRELOAD: 
                startAutoUpdate();
                break;
            case LOAD_ON_DEMAND: 
                removeBinaries();
                startAutoUpdate();
                break;
        }
    }
    
    /**
     * Sets the binary cache root directory
     *
     * @param path the absolute path of the directory containing
     * files for this cache.
     */
    public void setPath(String path) {
        setDirectory(new File(path));
    }

    /**
     * Returns the binary cache root directory
     *
     * @return the absolute path of the directory containing
     * files for this cache.
     */
    public String getPath() {
        return cacheDir.toString();
    }

    /**
     * Sets the binary cache root directory
     *
     * @param dir the root directory containing files for this cache
     */
    public synchronized void setDirectory(File dir) {
        removeBinaries();
        this.cacheDir = dir;
    }

    /**
     * Returns the binary cache root directory
     *
     * @return the root directory containing files for this cache
     */
    public File getDirectory() {
        return cacheDir;
    }

    /**
     * Sets the interval between two cache updates. This is meaningful
     * only if the cache policy is set to LOAD_ON_DEMAND or PRELOAD.
     * Defaults to five seconds.
     *
     * @param delay the number of seconds between cache updates
     */
    public synchronized void setDelay(long delay) {
        this.delay = delay * 1000;
    }

    /**
     * Returns the interval between two cache updates. This is meaningful
     * only if the cache policy is set to LOAD_ON_DEMAND or PRELOAD.
     *
     * @return the number of seconds between cache updates
     */
    public long getDelay() {
        return delay / 1000;
    }

    /**
     * Sets the file suffix. If set, files that do not have this suffix 
     * will be ignored when read into the cache.
     *
     * @param filenameSuffix the optional filename suffix of 
     * files to be read for this cache.  
     */
    public void setFilenameSuffix(String filenameSuffix) {
        this.filenameSuffix = filenameSuffix;
    }

    /**
     * Returns the file suffix. If set, files that do not have this suffix 
     * will be ignored when read into the cache.
     *
     * @param the optional filename suffix of 
     * files to be read for this cache.  
     */
    public String getFilenameSuffix() {
        return filenameSuffix;
    }

    public void addCacheListener(CacheListener listener) {
        multicaster.addListener(listener);
    }

    public void removeCacheListener(CacheListener listener) {
        multicaster.removeListener(listener);
    }

    /**
     * Gets a binary object from the cache
     *
     * @param name binary object's filename, including its
     * path relative to the cache's root directory.
     *
     * @return the corresponding <tt>BinaryData</tt>, or null
     * if not found or an error has occurred
     */
    public synchronized BinaryData getBinaryData(String name) {
        if (!(filenameSuffix == null || name.endsWith(filenameSuffix))) {
            fireCacheEvent(updateFailedAdapter, name,
                    new IOException("The requested name, \"" + name +
                    "\", does not have the filename suffix \"" +
                    filenameSuffix + "\"."));
            return null;
        }
                
    /* 
     * If we're preloading, then we can just return the file from the cache
     * if we have it -- it will be updated the next time our update() method is
     * called. Similarly, if we're loading on an ad hoc basis the file will 
     * be updated the next time our update(String name) method is called. On 
     * the other hand if we're loading on the demand, we have to check if 
     * the file has been changed since we last  cached it.
     */
        switch (loadingPolicy) {
            case LOAD_AD_HOC:
            case PRELOAD: 
                CacheElement element = (CacheElement)cache.get(name);
                return element != null ? (BinaryData)element.object : null;
            case LOAD_ON_DEMAND: // If we're loading on demand, check the file.
                if (checkCacheDir())
                    return doUpdate(name);
            default:
                return null;
        } 
    }

    public void startAutoUpdate() {
        if(loadingPolicy != LOAD_AD_HOC) {
            synchronized (this) {
                stopAutoUpdate();
                if (timer == null) {
                    timer = new UpdateTimer(this, delay);
                }
                update();
                timer.startTiming();
            }
        }
    }

    public void stopAutoUpdate() {
        if (timer != null) {
            timer.stopTiming();
            timer = null;
        }
    }

    /**
     * Returns a list of cached files
     * @return a list of cached files
     */
    public Iterator listCachedFiles() {
        return Collections.unmodifiableCollection(cache.values()).iterator();
    }

    /**
     * Tells whether the binary cache root directory exists and is readable
     *
     * @return true if the cache directory exists and is readable;
     * otherwise, also fires an appropriate <tt>CacheEvent</tt>.
     */
    private boolean checkCacheDir() {
        if (!cacheDir.exists()) {
            fireCacheEvent(unavailableAdapter, null, 
                    new IOException("Directory \"" +
                    cacheDir.getAbsolutePath() + "\" not found."));
            stopAutoUpdate();
            return false;
        }
        if (!cacheDir.isDirectory()) {
            fireCacheEvent(unavailableAdapter, null,
                   new IOException("\"" + cacheDir.getAbsolutePath() +
                   "\" is not a directory."));
            stopAutoUpdate();
            return false;
        }
        return true;
    }

    /**
     * Loads the binary object from a file
     *
     * @param file file to read from
     * @return binary object
     */
    private BinaryData loadBinaryData(File file) throws IOException {
        return new BinaryData(file);
    }

    /**
     * Given a file, returns the BinaryData object from the file if it's more
     * recent then the one in the cache; otherwise, returns the one
     * in the cache.
     *
     * @param elementName the name the file should have in the cache
     * @param file the file from which the binary object may be read
     */
    private BinaryData getLatestBinaryData(String elementName, File file) {

        BinaryData bindata = null;
        long lastModified = file.lastModified();
        CacheElement element = (CacheElement)cache.get(elementName);

        try {
            // If we don't have this binary object, add it.
            if (element == null) {
                bindata = loadBinaryData(file);
                bindata.setBinaryCache(this);
                cache.put(elementName, 
                        new CacheElement(elementName, bindata, lastModified));
                fireCacheEvent(updatedAdapter, elementName, null);

            // If we have it and it's been modified, update it.
            } else if (lastModified > element.lastModified) {
                bindata = loadBinaryData(file);
                bindata.setBinaryCache(this);
                element.object = bindata;
                element.lastModified = lastModified;
                fireCacheEvent(updatedAdapter, elementName, null);
            } else {
            // Return the BinaryData object we already have.
                bindata = (BinaryData)element.object;
            }
        } catch (IOException e) {
            fireCacheEvent(updateFailedAdapter, elementName, e);
        }
        return bindata;
    }

    /**
     * Update a named binary object if in the LOAD_AD_HOC mode
     * Do nothing if in other modes
     * @param name of the binary object to update
     */
	public synchronized void update(String name) {
		if (loadingPolicy == FileBinaryCache.LOAD_AD_HOC) {
			if (checkCacheDir()) 
                doUpdate(name);
        }
	}

    /**
     * Translates the name of the binary object to its file representation,
     * checks for the latter's existence and makes sure it's not a directory
     *
     * @param name binary object's name
     * @return updated binary object
     */
	private BinaryData doUpdate(String name) {

		File file = nameToFile(name);

		if (!file.exists()) {
            removeBinaryData(name);
			fireCacheEvent(updateFailedAdapter, name, 
				new IOException("\""+file.getPath()+"\" doesn't exist"));
				return null;
		}
			
		if (!file.isFile()) {
			fireCacheEvent(updateFailedAdapter, name, 
				new IOException("\""+file.getAbsolutePath()+
                    "\" is a directory"));
			return null;
		}
			
		return getLatestBinaryData(name, file);
	}

    /**
     * Updates the cache. In LOAD_AD_HOC mode, this does nothing.
     */
    public synchronized void update() {
        if (!checkCacheDir()) {
            return;
        }
        switch (loadingPolicy) {
            case LOAD_AD_HOC:
                // Do nothing, ad hoc is not auto updated
                break;
            case LOAD_ON_DEMAND:
                removeDeletedBinariess();
                break;
            case PRELOAD:
                loadBinaries();
                break;
            default:
                break;
        }
    }

    /**
     * Load all files and remove the ones in the cache
     */
    private void loadBinaries() {
        LinkedList visitedFiles = new LinkedList();
        try {
            readDirectory(cacheDir, "", visitedFiles);
            removeUnvisitedFiles(visitedFiles);
        } catch (IOException e) {
            stopAutoUpdate();
            fireCacheEvent(unavailableAdapter, null, e);
            return;
        }
    }

    /**
     * Remove all binary objects from the cache
     */
    private void removeBinaries() {
        cache.clear();
    }

    /**
     * Remove from the cache any objects corresponding to files we
     * didn't just visit.
     * @param list of visited objects
     */
    private void removeUnvisitedFiles(LinkedList visitedFiles) {

        Set keySet = cache.keySet();
        Iterator keyIterator = keySet.iterator();

        while (keyIterator.hasNext()) {
            String elementName = (String)keyIterator.next();
            if (!visitedFiles.contains(elementName)) {
                keyIterator.remove();
                fireCacheEvent(removedAdapter, elementName, null);
            }
        } 
    }
    /**
     * Removes from the cache binary objects that correspond to deleted files.
     */
    private void removeDeletedBinariess() {

        Set keySet = cache.keySet();
        Iterator keyIterator = keySet.iterator();

        while (keyIterator.hasNext()) {
           String elementName = (String)keyIterator.next();
           File file = nameToFile(elementName);
           if (!file.exists()) {
                keyIterator.remove();
                fireCacheEvent(removedAdapter, elementName, null);
           }
        }
    }                

    /**
     * Removes named binary object from the cache.
     * @param name of the file corresponding to the binary object in the cache
     */
    private void removeBinaryData(String name) {
        cache.remove(name);
        fireCacheEvent(removedAdapter, name, null);
    } 

    /**
     * Recursively updates the cache from the files in a (sub)directory
     * and its subdirectories.
     *
     * @param dir the directory to be read.
     * @param relativeDirPath a string representing the directory's path
     * relative to the root cache directory.
     * @param visitedFiles a List of files that have been visited so far.
     */
    private void readDirectory(File dir, String relativeDirPath, 
            List visitedFiles) throws IOException {
        String[] filenames = dir.list();
        if (filenames == null) {
            throw new IOException("Could not get file list from directory \"" 
                + dir.getAbsolutePath() + "\".");
        }
        // Iterate through the items in the directory.
        for (int fileNum = 0; fileNum < filenames.length; fileNum++) {
            String filename = filenames[fileNum];
            File file = new File(dir, filename);
            String elementName = relativeDirPath + filename;
            // If the item is a file, see if we need to to read it.
            if (file.isFile()) {
            // If we have no filename suffix, or if we have one and this
            // file ends with it, check the file.
                if (filenameSuffix == null || 
                        filename.endsWith(filenameSuffix)) {
                    visitedFiles.add(elementName);
                    getLatestBinaryData(elementName, file);
                }
            } else if (file.isDirectory()) {
            // If the item is a directory, recursively read it.
                readDirectory(file, elementName + "/", visitedFiles);
            }
        } 
    }

    /**
     * Converts a cache element name to a <tt>File</tt>.
     */
    private File nameToFile(String name) {
        // Replace forward slashes with the operating system's
        // file separator, if it's not a forward slash.
        if (File.separatorChar == '/') {
            return new File(cacheDir, name);
        } else {
            StringBuffer pathBuf = new StringBuffer();
            for (int i = 0; i < name.length(); i++) {
                char c = name.charAt(i);
                if (c == '/') {
                    pathBuf.append(File.separatorChar);
                } else {
                    pathBuf.append(c);
                }
            }
            return new File(cacheDir, pathBuf.toString());
        }
    }

    /**
     * A convenience method for firing a CacheEvent.
     *
     * @param adapter a <tt>ListenerAdapter</tt>.
     * @param elementName the name of the cache element in question, or null.
     * @param e an <tt>Exception</tt> to be included with the event, or null.
     */
    private void fireCacheEvent(ListenerAdapter adapter, String elementName, 
            Exception e) {
        CacheEvent event = new CacheEvent(this);
        event.setElementName(elementName);
        event.setException(e);
        multicaster.fireEvent(event, adapter);
    }
}
