/*
 * FreeMarker: a tool that allows Java programs to generate HTML
 * output using templates.
 * Copyright (C) 1998 Benjamin Geer
 * Email: beroul@yahoo.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

package freemarker.template;

import freemarker.template.compiler.*;
import freemarker.template.instruction.*;
import java.io.*;
import java.util.*;

/**
 * <p>An application or servlet can instantiate a subclass of <tt>BinaryData</tt>
 * to retrieve a binary file.
 *
 * <p>You can pass the filename of the binary file to the constructor, in
 * which case it is read in immediately.
 *
 * <p>To retrive the binary data, call the <tt>process()</tt> method. 
 *
 * <p>To facilitate multithreading, <tt>BinaryData</tt> objects are
 * immutable; if you need to reload a binary file, you must make a new
 * <tt>BinaryData</tt> object.  In most cases, it will be sufficient
 * to let a <tt>BinaryCache</tt> do this for you.
 *
 * @see freemarker.template.BinaryCache
 */
 
public class BinaryData {
	private byte[] dataarray = null;
	private BinaryCache cache;
	private Map functions = new HashMap();

    /**
     * Constructs an empty template.
     */
    public BinaryData() { }

    /**
     * Constructs a binary data object by compiling it from a file.  Calls
     * <tt>compileFromFile()</tt>.
     *
     * @param filePath the absolute path of the binary file
     * to be compiled.
     */
    public BinaryData(String filePath) throws IOException {
		compileFromFile(filePath);
    }

    /**
     * Constructs a BinaryData object by compiling it from a file.  Calls
     * <tt>compileFromFile()</tt>.
     *
     * @param file a <tt>File</tt> representing the
     * binary file to be compiled.
     */
    public BinaryData(File file) throws IOException {
		compileFromFile(file);
    }

    /**
     * Constructs a template by compiling it from an
     * <tt>InputStream</tt>.  Calls
     * <tt>compileFromStream()</tt>.
     *
     * @param stream an <tt>InputStream</tt> from which the
     * template can be read.
     */
    public BinaryData(InputStream stream) throws IOException {
		compileFromStream(stream);
    }

    /**
     * Reads and compiles a template from a file, by getting the file's
     * <tt>FileInputStream</tt> and using it to call <tt>compileFromStream()</tt>, using
     * the platform's default character encoding.
     *
     * @param filePath the absolute path of the template file
     * to be compiled.
     */
    public synchronized void compileFromFile(String filePath) throws IOException {
		File file = new File(filePath);
		compileFromFile(file);
    }

    /**
     * Reads and compiles a template from a file, by getting the file's
     * <tt>FileInputStream</tt> and using it to call <tt>compileFromStream()</tt>, using
     * the platform's default character encoding.
     *
     * @param file a <tt>File</tt> representing the
     * template file to be compiled.
     */
    public synchronized void compileFromFile(File file) throws IOException {
        FileInputStream inputStream;

		if (!file.exists()) {
			throw new FileNotFoundException("Template file " +
				file.getName() +
				" not found.");
		}
		if (!file.canRead()) {
			throw new IOException("Can't read from template file " +
				file.getName() +
				".");
		}

        inputStream = new FileInputStream(file);
        compileFromStream(inputStream);
        inputStream.close();
    }

    /**
     * Compiles the template from an <tt>InputStream</tt>, using the platform's default
	 * character encoding.  If the template has already been compiled, this method does
	 * nothing.
     *
     * @param stream an <tt>InputStream</tt> from which the
     * template can be read.
     */
    public synchronized void compileFromStream(InputStream stream) throws IOException {
		ByteArrayOutputStream arraystream = new ByteArrayOutputStream(8192);
		byte[] buffer = new byte[4096];
		int bytes_read;
		
		while ( (bytes_read = stream.read(buffer)) != -1) {
			arraystream.write( buffer, 0, bytes_read );
		}
		arraystream.close();
		stream.close();
		
		dataarray = arraystream.toByteArray();
	}

	/**
	 * Sets the <tt>BinaryCache</tt> that this file belongs to.
	 * <tt>IncludeInstruction</tt> objects will be able to request this
	 * <tt>BinaryCache</tt> at run-time.
	 *
	 * @param cache the <tt>BinaryCache</tt> that this binary data file belongs to.
	 */
	public void setBinaryCache(BinaryCache cache) {
		this.cache = cache;
	}

	/**
	 * Gets the <tt>BinaryCache</tt> that this binary data file belongs to.
	 *
	 * @return the BinaryCache that this binary data file belongs to.
	 */
	public BinaryCache getBinaryCache() {
		return cache;
	}

    /**
     * Processes the binary data file, and output 
     * the resulting binary data to an <tt>OutputStream</tt>.
     *
     * @param modelRoot the root node of the data model.  If null, an
	 * empty data model is used.
     * @param out an <tt>OutputStream</tt> to output the HTML to.
     */
    public void process(OutputStream out) {
		try {
			if (dataarray != null) {
				out.write( dataarray );
			}
		}
		catch (IOException e) {	}
    }

    /**
     * Used by FreeMarker classes to format an error message as an HTML comment.
     */
    public static String formatErrorMessage(String errorMessage) {
		return "BinaryData Error: " + errorMessage;
    }

    /**
     * Used by FreeMarker classes to format a stack trace as a string.
     */
    public static String getStackTrace(Exception e) {
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);
		e.printStackTrace(pw);
		pw.close();
		return sw.toString();
    }
}
