package guestbook;

import java.io.*;
import javax.servlet.*;
import javax.servlet.http.*;
import freemarker.template.*;

public class GuestbookServlet extends HttpServlet implements CacheListener {

    private Guestbook book;
    private FileTemplateCache templateCache;
    private String templateSuffix;

    public void init(ServletConfig config) throws ServletException {
        super.init(config);

        String templatePath = getInitParameter("templatePath");
        if (templatePath == null) {
            throw new UnavailableException(this, "Init parameter templatePath not set.");
        }

        templateSuffix = getInitParameter("templateSuffix");
        if (templateSuffix == null) {
            throw new UnavailableException(this, "Init parameter templateSuffix not set.");
        }

        String realPath = getServletContext().getRealPath(templatePath);
        templateCache = new FileTemplateCache(realPath);
        templateCache.setFilenameSuffix(templateSuffix);
        templateCache.addCacheListener(this);
		templateCache.startAutoUpdate();
        book = new Guestbook();
    }

    public void doGet(HttpServletRequest req, HttpServletResponse res)
        throws ServletException, IOException {

        res.setContentType("text/html");
        PrintWriter out = new PrintWriter(res.getOutputStream());

        // Get the guestbook's template.
        Template template = getTemplate("guestbook" + templateSuffix);

        // Make the root node of the data model.
        SimpleHash modelRoot = new SimpleHash();

        // If an entry was submitted, add it to the guestbook.
        if (req.getParameter("dataSubmitted") != null) {
            book.addEntry(req.getParameter("guestName"), req.getParameter("message"));
            modelRoot.put("entryAdded", new SimpleScalar(true));
        }

        // Wrap the guestbook in a template model adapter.
        GuestbookTM bookTM = new GuestbookTM(book);
        modelRoot.put("guestbook", bookTM);

        // Process the template.
        template.process(modelRoot, out);

        out.close();
    }

    public void doPost(HttpServletRequest req, HttpServletResponse res)
        throws ServletException, IOException {
        doGet(req, res);
    }

    private Template getTemplate(String templateName) throws ServletException {
        Template template = templateCache.getTemplate(templateName);
        if (template == null) {
            throw new ServletException("Template " +
				       templateName +
				       " is not available.");
        } else {
            return template;
        }
    }

    public void cacheUnavailable(CacheEvent e) {
        System.out.println("Cache unavailable: " + e.getException().toString());
    }

    public void elementUpdated(CacheEvent e) {
        System.out.println("Template updated: " + e.getElementName());
    }

    public void elementUpdateFailed(CacheEvent e) {
        System.out.println("Update of template " + e.getElementName() +
			   " failed: " + e.getException().toString());
    }

    public void elementRemoved(CacheEvent e) {
        System.out.println("Template removed: " + e.getElementName());
    }

    public String getServletInfo() {
        return "Guestbook Servlet";
    }
}
