/*
 * ====================================================================
 *
 * Copyright (c) 2000 Attila Szegedi.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes the Expose library
 *        (http://www.szegedi.org/expose)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Expose" and "Attila Szegedi" must not be used to endorse 
 *    or promote products derived from this software without prior written
 *    permission. For written permission, please contact expose@szegedi.org.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR ITS CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 */

package org.szegedi.expose.servlet;

import freemarker.template.*;

import java.io.*;
import java.util.*;
import java.text.*;

import javax.servlet.*;
import javax.servlet.http.*;

/**
 * This is a general-purpose Freemarker servlet with following features:
 * <ul>
 * <li>It utilizes a TemplateCache that is fully parametrizable</li>
 * <li>It makes all request, request parameters, session, and servlet 
 * context attributes available to templates through "Request", "RequestParameters",
 * "Session", and "Application" variables. The implementation for these variables
 * is efficient; rather than copying the attributes and parameters into <tt>SimpleHash</tt>, 
 * special wrapper classes are used that implement the {@link TemplateHashModel}
 * interface.</li>
 * </ul>
 * Supported initialization parameters are:
 * <ul>
 * <li><strong>TemplatePath</strong> specifies the location of the templates.
 * By default, the location is the web application-relative URI of
 * the location. Alternatively, you can prepend it with <tt>file://</tt> to
 * indicate path in the file system (i.e. <tt>file:///var/www/project/templates/</tt>).
 * Note that three slashes were specified.
 * Default value is <tt>/</tt> (that is, the root of the web application).</li>
 * <li><strong>TemplateUpdateInterval</strong> value for update interval of the
 * template cache in milliseconds. Default is <tt>60000</tt> (ten minutes)</li>
 * <li><strong>NoCache</strong> if set to true, generates headers in the response
 * that advise the HTTP client not to cache the returned page. Default <tt>false</tt></li>
 * </ul>
 * @author Attila Szegedi, attila@szegedi.org
 * @version 1.0
 */

public class ExposeServlet
extends
	HttpServlet
{
	public static final long serialVersionUID = -2440216393145762479L;

	private static final String INITPARAM_TEMPLATE_PATH = "TemplatePath";
	private static final String INITPARAM_TEMPLATE_PATH_DEFAULT = "/";
	private static final String INITPARAM_TEMPLATE_DELAY = "TemplateUpdateInterval";
	private static final String INITPARAM_TEMPLATE_DELAY_DEFAULT = "60000";
	private static final String INITPARAM_NOCACHE = "NoCache";
	
	private static final String KEY_REQUEST = "Request";
	private static final String KEY_REQUEST_PARAMETERS = "RequestParameters";
	private static final String KEY_SESSION = "Session";
	private static final String KEY_APPLICATION = "Application";
	
	// Note these names start with dot, so they're essentially invisible from
	// a freemarker script.
	private static final String ATTR_REQUEST_MODEL = ".freemarker.Request";
	private static final String ATTR_REQUEST_PARAMETERS_MODEL = ".freemarker.RequestParameters";
	private static final String ATTR_SESSION_MODEL = ".freemarker.Session";
	private static final String ATTR_APPLICATION_MODEL = ".freemarker.Application";
	
	private static final String EXPIRATION_DATE;
	
	static
	{
		// Generate expiration date that is one year from now in the past
		GregorianCalendar expiration = new GregorianCalendar();
		expiration.roll(Calendar.YEAR, -1);
		SimpleDateFormat httpDate = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss z", java.util.Locale.US);
		EXPIRATION_DATE = httpDate.format(expiration.getTime());
	}
	
	private TemplateCache cache;
	private boolean nocache;
	protected boolean debug;
	
	public void init()
	{
		cache = createCache();
		// Set browser caching policy
		String strnocache = getServletConfig().getInitParameter(INITPARAM_NOCACHE);
		nocache = ("true".equalsIgnoreCase(strnocache) || "yes".equalsIgnoreCase(strnocache));

		String strdebug = getServletConfig().getInitParameter("debug");
		debug = ("true".equalsIgnoreCase(strdebug) || "yes".equalsIgnoreCase(strdebug));
	}
	
	protected TemplateCache createCache()
	{
		String templatePath = getServletConfig().getInitParameter(INITPARAM_TEMPLATE_PATH);
		String strUpdateInterval = getServletConfig().getInitParameter(INITPARAM_TEMPLATE_DELAY);
		if(templatePath == null)
			templatePath = "/";
		long updateInterval = 10 * 60 * 1000L;
		try
		{
			if(strUpdateInterval != null)
			updateInterval = Long.parseLong(strUpdateInterval);
		}
		catch(NumberFormatException e)
		{
			// Intentionally ignored
		}
		// See if filesystem path is specified
		if(templatePath.startsWith("file://"))
			templatePath = templatePath.substring(7);
		// if not, resolve it according to the location of servlet context.
		else
			templatePath = getServletContext().getRealPath(templatePath);
		if(debug)
			System.err.println("TemplatePath: " + templatePath);
		return new FileTemplateCache(templatePath, updateInterval);
	}
	
	public void doGet(HttpServletRequest request, HttpServletResponse response)
	throws
		ServletException,
		IOException
	{
		process(request, response);
	}

	public void doPost(HttpServletRequest request, HttpServletResponse response)
	throws
		ServletException,
		IOException
	{
		process(request, response);
	}

	private void process(HttpServletRequest request, HttpServletResponse response)
	throws
		ServletException,
		IOException
	{
		String path = request.getPathInfo();
		if(path == null) path = "";
		
		if(debug)
			System.err.println("Requested template: " + path);
		
		Template template = cache.getTemplate(path);
		if(template == null)
		{
			response.sendError(HttpServletResponse.SC_NOT_FOUND);
			return;
		}
		
		// Set cache policy
		setBrowserCachingPolicy(response);

		SimpleHash params = new SimpleHash();
		
		// Create hash model wrapper for servlet context (the application)
		ServletContext servletContext = getServletContext();
		ServletContextHashModel servletContextModel = (ServletContextHashModel)servletContext.getAttribute(ATTR_APPLICATION_MODEL);
		if(servletContextModel == null)
		{
			servletContextModel = new ServletContextHashModel(servletContext);
			servletContext.setAttribute(ATTR_APPLICATION_MODEL, servletContextModel);
			initializeServletContext(request, response);
		}
		params.put(KEY_APPLICATION, servletContextModel);
		
		// Create hash model wrapper for session
		HttpSession session = request.getSession();
		HttpSessionHashModel sessionModel = (HttpSessionHashModel)session.getAttribute(ATTR_SESSION_MODEL);
		if(sessionModel == null)
		{
			sessionModel = new HttpSessionHashModel(session);
			session.setAttribute(ATTR_SESSION_MODEL, sessionModel);
			initializeSession(request, response);
		}
		params.put(KEY_SESSION, sessionModel);
		
		// Create hash model wrapper for request
		HttpRequestHashModel requestModel = (HttpRequestHashModel)request.getAttribute(ATTR_REQUEST_MODEL);
		if(requestModel == null)
		{
			requestModel = new HttpRequestHashModel(request);
			request.setAttribute(ATTR_REQUEST_MODEL, requestModel);
		}
		params.put(KEY_REQUEST, requestModel);
		
		// Create hash model wrapper for request parameters
		HttpRequestParametersHashModel requestParametersModel = (HttpRequestParametersHashModel)request.getAttribute(ATTR_REQUEST_PARAMETERS_MODEL);
		if(requestParametersModel == null)
		{
			requestParametersModel = createRequestParametersHashModel(request);
			request.setAttribute(ATTR_REQUEST_PARAMETERS_MODEL, requestParametersModel);
		}
		params.put(KEY_REQUEST_PARAMETERS, requestParametersModel);
		
		// Give subclasses a chance to hook into preprocessing
		if(preTemplateProcess(request, response, template, params))
		{
			try
			{
				// Process the template
				template.process(params, response.getWriter());
			}
			finally
			{
				// Give subclasses a chance to hook into postprocessing
				postTemplateProcess(request, response, template, params);
			}
		}
	}

	protected HttpRequestParametersHashModel createRequestParametersHashModel(HttpServletRequest request)
	{
		return new HttpRequestParametersHashModel(request);
	}
	
	/**
	 * Called when servlet detects in a request processing that 
	 * application-global (that is, ServletContext-specific) attributes are not yet 
	 * set.
	 * This is a generic hook you might use in subclasses to perform a specific 
	 * action on first request in the context. By default it does nothing.
	 * @param request the actual HTTP request
	 * @param response the actual HTTP response
	 */
	protected void initializeServletContext(HttpServletRequest request, HttpServletResponse response)
	throws
		ServletException,
		IOException
	{
	}

	/**
	 * Called when servlet detects in a request processing that 
	 * session-global (that is, HttpSession-specific) attributes are not yet 
	 * set.
	 * This is a generic hook you might use in subclasses to perform a specific 
	 * action on first request in the session. By default it does nothing.
	 * @param request the actual HTTP request
	 * @param response the actual HTTP response
	 */
	protected void initializeSession(HttpServletRequest request, HttpServletResponse response)
	throws
		ServletException,
		IOException
	{
	}

	/**
	 * Called before the execution is passed to template.process().
	 * This is a generic hook you might use in subclasses to perform a specific 
	 * action before the template is processed. By default does nothing.
	 * A typical action to perform here is to inject application-specific
	 * objects into the <tt>TemplateModelRoot</tt>.
	 * @param request the actual HTTP request
	 * @param response the actual HTTP response
	 * @param template the template that will get executed
	 * @param data the data that will be passed to the template
	 * @return true to process the template, false to suppress template processing.
	 */
	protected boolean preTemplateProcess(HttpServletRequest request, HttpServletResponse response, Template template, TemplateModelRoot data)
	throws
		ServletException,
		IOException
	{
		return true;
	}
	
	/**
	 * Called after the execution returns from template.process().
	 * This is a generic hook you might use in subclasses to perform a specific 
	 * action after the template is processed. It will be invoked even if the 
	 * template processing throws an exception. By default does nothing.
	 * @param request the actual HTTP request
	 * @param response the actual HTTP response
	 * @param template the template that was executed
	 * @param data the data that was passed to the template
	 */
	protected void postTemplateProcess(HttpServletRequest request, HttpServletResponse response, Template template, TemplateModelRoot data)
	throws
		ServletException,
		IOException
	{
	}

	/**
	 * If the parameter "nocache" was set to true, generate a set of headers
	 * that will advise the HTTP client not to cache the returned page.
	 */
	private void setBrowserCachingPolicy(HttpServletResponse response)
	{
		if(nocache)
		{
			// HTTP 1.1 browsers should defeat caching on this header
			response.setHeader("Cache-Control", "no-cache");
			// HTTP 1.0 browsers should defeat caching on this header
			response.setHeader("Pragma", "no-cache");
			// Last resort for those that ignore all of the above
			response.setHeader("Expires", EXPIRATION_DATE);
		}
	}
}
