/*
 * ====================================================================
 *
 * Copyright (c) 2000 Attila Szegedi.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes the Expose library
 *        (http://www.szegedi.org/expose)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Expose" and "Attila Szegedi" must not be used to endorse 
 *    or promote products derived from this software without prior written
 *    permission. For written permission, please contact expose@szegedi.org.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR ITS CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 */

package org.szegedi.expose.model;

import org.szegedi.collection.IdentityHashMap;

import freemarker.template.*;

import java.lang.ref.*;
import java.lang.reflect.*;
import java.util.*;

/**
 * Wraps the static fields and methods of a class in a {@link TemplateHashModel}.
 * Fields are wrapped using {@link ReflectionUtilities#wrap(Object)}, and
 * methods are wrapped into {@link ReflectionMethodModel} instances.
 * Unfortunately, there is currently no support for bean property-style
 * calls of static methods, as in {@link ReflectionObjectModel}.
 * @author Attila Szegedi, attila@szegedi.org
 * @version 1.0
 */
public final class StaticModel
implements
	TemplateHashModel
{
	private static final Map modelCache = new IdentityHashMap();
	private static final ReferenceQueue refQueue = new ReferenceQueue();

	private Class clazz;
	private final Map map = new HashMap();
		
	private StaticModel(Class clazz)
	{
		this.clazz = clazz;
		populate();
	}
	
	/**
	 * Creates a model representing the static fields and
	 * methods of the passed class. Models are cached, meaning
	 * that calling create twice with a same class parameter
	 * will yield the same object.
	 */
	public static final StaticModel create(Class clazz)
	{
		StaticModel model = null;
		ModelReference ref = null;
		// NOTE: we're doing minimal synchronizations -- which can lead to
		// duplicate wrapper creation. However, this has no harmful side-effects and
		// is a lesser performance hit.
		synchronized(modelCache)
		{
			ref = (ModelReference)modelCache.get(clazz);
		}

		if(ref != null)
		{
			model = ref.getModel();
			if(model != null)
				return model;
		}

		// Create new model if none found
		model = new StaticModel(clazz);
		synchronized(modelCache)
		{
			// Remove cleared references
			for(;;)
			{
				ModelReference queuedRef = (ModelReference)refQueue.poll();
				if(queuedRef == null)
					break;
				modelCache.remove(queuedRef.clazz);
			}
			// Register new reference
			modelCache.put(clazz, new ModelReference(model));
		}
		return model;	
	}

	/**
	 * Returns the field or method named by the <tt>key</tt>
	 * parameter.
	 */
	public TemplateModel get(String key)
	throws
		TemplateModelException
	{
		Object model = map.get(key);
		// Method or final field -- these have cached template models
		if(model instanceof TemplateModel)
			return (TemplateModel)model;
		// Non-final field; this must be evaluated on each call.
		if(model instanceof Field)
		{
			try
			{
				return ReflectionUtilities.wrap(((Field)model).get(null));
			}
			catch(IllegalAccessException e)
			{
				throw new TemplateModelException("Illegal access for field " + key + " of class " + clazz.getName());
			}
		}

		throw new TemplateModelException("No such key: " + key + " in class " + clazz.getName());
	}

	/**
	 * Returns true if there is at least one public static
	 * field or method in the underlying class.
	 */
	public boolean isEmpty()
	{
		return map.isEmpty();
	}

	private void populate()
	{
		Field[] fields = clazz.getFields();
		for(int i = 0; i < fields.length; ++i)
		{
			Field field = fields[i];
			int mod = field.getModifiers();
			if(Modifier.isPublic(mod) && Modifier.isStatic(mod))
			{
				if(Modifier.isFinal(mod))
					try
					{
						map.put(field.getName(), ReflectionUtilities.wrap(field.get(null)));
					}
					catch(IllegalAccessException e)
					{
						// Intentionally ignored
					}
				else
					// This is a special flagging value: Field in the map means
					// that this is a non-final field, and it must be evaluated
					// on each get() call.
					map.put(field.getName(), field);
			}
		}
		Method[] methods = clazz.getMethods();
		for(int i = 0; i < methods.length; ++i)
		{
			Method method = methods[i];
			int mod = method.getModifiers();
			if(Modifier.isPublic(mod) && Modifier.isStatic(mod))
			{
				try
				{
					method = ReflectionUtilities.getPublicMethod(method);
				}
				catch(NoSuchMethodException e)
				{
					// Intentionally ignored
				}
				map.put(method.getName(), new ReflectionMethodModel(null, method));
			}
		}
	}
	
	/**
	 * A special weak reference that is registered in the modelCache.
	 * When it gets cleared (that is, the model became unreachable)
	 * it will remove itself from the model cache.
	 */
	private static final class ModelReference extends WeakReference
	{
		Class clazz;

		ModelReference(StaticModel ref)
		{
			super(ref, refQueue);
			clazz = ref.clazz;
		}
		
		StaticModel getModel()
		{
			return (StaticModel)this.get();
		}
	}
}
