/*
 * ====================================================================
 *
 * Copyright (c) 2000 Attila Szegedi.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes the Expose library
 *        (http://www.szegedi.org/expose)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Expose" and "Attila Szegedi" must not be used to endorse 
 *    or promote products derived from this software without prior written
 *    permission. For written permission, please contact expose@szegedi.org.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR ITS CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 */

package org.szegedi.expose.model;

import freemarker.template.*;

import java.text.*;
import java.util.*;

/**
 * A hash model that wraps a resource bundle. Makes it convenient to store
 * localized content in the data model. It also acts as a method model that will
 * take a resource key and arbitrary number of arguments and will apply
 * {@link MessageFormat} with arguments on the string represented by the key.
 * Typical usages:
 * <ul>
 * <li><tt>bundle.resourceKey</tt> will retrieve the object from resource bundle
 * with key <tt>resourceKey</tt></li>
 * <li><tt>bundle("patternKey", arg1, arg2, arg3)</tt> will retrieve the string
 * from resource bundle with key <tt>patternKey</tt>, and will use it as a pattern
 * for MessageFormat with arguments arg1, arg2 and arg3</li>
 * </ul>
 * @author Attila Szegedi, attila@szegedi.org
 * @version 1.0
 */
public final class ResourceBundleHashModel
extends
	ReflectionObjectModelBase
implements
	TemplateHashModel,
	TemplateMethodModel
{
	private Hashtable formats = null;

	public ResourceBundleHashModel(ResourceBundle bundle)
	{
		super(bundle);
	}

	/**
	 * Returns a model wrapping the specified resource bundle object. If there is already
	 * a cached model instance for this resource bundle, returns the cached model instance.
	 * Models are cached using {@link WeakReference} objects. The caching can be turned
	 * off by setting the <tt>expose.reflection.nocache</tt> system property to
	 * true. In this case calling this method is equivalent to constructing a new model.
	 * @param object the resource bundle to wrap into a model.
	 * @return the model for the resource bundle
	 */
	public static final ResourceBundleHashModel getInstance(ResourceBundle object)
	{
		if(noCache)
			return new ResourceBundleHashModel(object);
		
		ResourceBundleHashModel model = (ResourceBundleHashModel)lookup(object);
		if(model == null)
		{
			model = new ResourceBundleHashModel(object);
			register(model, object);
		}
		return model;
	}
	
	/**
	 * Returns the type of this object (which is TYPE_RESOURCE_BUNDLE)
	 */
	public int getType()
	{
		return TYPE_RESOURCE_BUNDLE;
	}

	/**
	 * Retrieves the object with specified key from resource bundle, and returns it
	 * wrapped in an appropriate TemplateModel.
	 */
	public TemplateModel get(String key)
	throws
		TemplateModelException
	{
		try
		{
			return ReflectionUtilities.wrap(((ResourceBundle)object).getObject(key), ReflectionUtilities.WRAP_AS_OBJECT);
		}
		catch(MissingResourceException e)
		{
			throw new TemplateModelException("No such key: " + key);
		}
	}

	/**
	 * Returns true if this bundle contains no objects.
	 */
	public boolean isEmpty()
	{
		return !((ResourceBundle)object).getKeys().hasMoreElements();
	}

	/**
	 * Takes first argument as a resource key, looks up a string in resource bundle
	 * with this key, then applies a MessageFormat.format on the string with the
	 * rest of the arguments. The created MessageFormats are cached for later reuse.
	 */
	public TemplateModel exec(List arguments)
	throws
		TemplateModelException
	{
		// Must have at least one argument - the key
		if(arguments.size() < 1)
			throw new TemplateModelException("No message key was specified");
		// Read it
		Iterator it = arguments.iterator();
		String key = it.next().toString();

		// Copy remaining arguments into an Object[]
		int args = arguments.size() - 1;
		Object[] params = new Object[args];
		for(int i = 0; i < args; ++i)
			params[i] = it.next();

		// Invoke format
		try
		{
			return new SimpleScalar(format(key, params));
		}
		catch(MissingResourceException e)
		{
			throw new TemplateModelException("No such key: " + key);
		}
	}

	/**
	 * Provides direct access to caching format engine from code (instead of from script).
	 */
	public String format(String key, Object[] params)
	throws
		MissingResourceException
	{
		// Check to see if we already have a cache for message formats
		// and construct it if we don't
		// NOTE: this block statement should be synchronized. However
		// concurrent creation of two caches will have no harmful
		// consequences, and we avoid a performance hit.
		/* synchronized(this) */
		{
			if(formats == null)
				formats = new Hashtable();
		}

		MessageFormat format = null;
		// Check to see if we already have a requested MessageFormat cached
		// and construct it if we don't
		// NOTE: this block statement should be synchronized. However
		// concurrent creation of two formats will have no harmful
		// consequences, and we avoid a performance hit.
		/* synchronized(formats) */
		{
			format = (MessageFormat)formats.get(key);
			if(format == null)
			{
				format = new MessageFormat(((ResourceBundle)object).getString(key));
				formats.put(key, format);
			}
		}

		// Perform the formatting
		return format.format(params);
	}
	
	public ResourceBundle getBundle()
	{
		return (ResourceBundle)object;
	}
}
