/*
 * ====================================================================
 *
 * Copyright (c) 2000 Attila Szegedi.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes the Expose library
 *        (http://www.szegedi.org/expose)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Expose" and "Attila Szegedi" must not be used to endorse 
 *    or promote products derived from this software without prior written
 *    permission. For written permission, please contact expose@szegedi.org.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR ITS CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 */

package org.szegedi.expose.model;

import freemarker.template.*;
import java.beans.*;
import java.lang.reflect.*;
import java.util.*;

/**
 * Utility class that provides generic services to reflection classes.
 * It handles all polymorphism issues in the {@link #wrap(Object)} and {@link #unwrap(Object)} methods.
 * @author Attila Szegedi, attila@szegedi.org
 * @version 1.0
 */
public final class ReflectionUtilities
{
	private static final Map publicMethodCache = new WeakHashMap();

	/**
	 * Use in the wrap() method to override logic for determining whether the
	 * parameter should be wrapped as a TemplateScalarModel or as a TemplateHashModel,
	 * and force wrapping into scalar.
	 */
	public static final boolean WRAP_AS_SCALAR = true;
	/**
	 * Use in the wrap() method to override logic for determining whether the
	 * parameter should be wrapped as a TemplateScalarModel or as a TemplateHashModel,
	 * and force wrapping into hash model.
	 */
	public static final boolean WRAP_AS_OBJECT = false;
			
	private ReflectionUtilities() {}
	
	/**
	 * Determines whether the object of this class should be wrapped into
	 * a {@link SimpleScalar} (true), or into a descendant of {@link ReflectionObjectModelBase} (false).
	 * All classes representing primitive types, as well as String and Boolean qualify.
	 */
	public static final boolean isScalar(Class clazz)
	{
		return clazz.isPrimitive() || clazz == java.lang.String.class || clazz == java.lang.Boolean.class;
	}
	
	/**
	 * Determines whether the object should be wrapped into a {@link SimpleScalar} 
	 * (true), or into a descendant of {@link ReflectionObjectModelBase} (false).
	 * Convenience method for {@link #isScalar(Class)} that deals with object being
	 * null.
	 * All classes representing primitive types, as well as String and Boolean qualify.
	 */
	public static final boolean isScalar(Object object)
	{
		return object == null || isScalar(object.getClass());
	}

	/**
	 * Wraps the object with a template model that is most specific for the object's
	 * class. Specifically:
	 * <ul>
	 * <li>if the object is null, returns {@link Models#EMPTY_SCALAR},</li>
	 * <li>if the object is a String returns a {@link SimpleScalar} for it,</li>
	 * <li>if the object is a Boolean returns a {@link BooleanScalar} for it,</li>
	 * <li>if the object is already a TemplateModel, returns it unchanged,</li>
	 * <li>if the object is an array, returns a {@link ReflectionArrayModel} for it
	 * <li>if the object is a Map, returns a {@link ReflectionMapModel} for it
	 * <li>if the object is a Collection, returns a {@link ReflectionCollectionModel} for it
	 * <li>if the object is an Iterator, returns a {@link ReflectionIteratorModel} for it
	 * <li>if the object is an Enumeration, returns a {@link ReflectionEnumerationModel} for it
	 * <li>otherwise, returns a {@link ReflectionObjectModel} for it
	 * </ul>
	 */
	public static final TemplateModel wrap(Object object)
	{
		return wrap(object, isScalar(object));
	}

	/**
	 * Wraps the object with a template model that is most specific for the object's
	 * class, just as {@link #wrap(Object)} would, however it can force wrapping into 
	 * scalar. (I.e. ReflectionMethodModel uses this when it knows that its return type 
	 * is a primitive type wrapper).
	 * @param object the object to wrap
	 * @param asScalar. If <tt>WRAP_AS_SCALAR</tt>, the object will be wrapped into
	 * a scalar (in case it is neither a Boolean nor a String, its <tt>toString()</tt>
	 * value is wrapped). If <tt>WRAP_AS_OBJECT</tt>, the object will be wrapped into
	 * {@link ReflectionObjectModel} even if it is a String or Boolean.
	 */
	public static final TemplateModel wrap(Object object, boolean asScalar)
	{
		if(object != null)
		{
			if(asScalar)
			{
				if(object instanceof Boolean)
					return BooleanScalar.getScalar(((Boolean)object).booleanValue());
				else
					return new SimpleScalar(object.toString());
			}
			else
			{
				if(object instanceof TemplateModel)
					return (TemplateModel)object;
				if(object.getClass().isArray())
					return ReflectionArrayModel.getInstance(object);
				if(object instanceof String)
					return ReflectionObjectModelBase.getInstance((String)object);
				if(object instanceof Map)
					return ReflectionMapModel.getInstance((Map)object);
				if(object instanceof Collection)
					return ReflectionCollectionModel.getInstance((Collection)object);
				if(object instanceof Iterator)
					return ReflectionIteratorModel.getInstance((Iterator)object);
				if(object instanceof Enumeration)
					return ReflectionEnumerationModel.getInstance((Enumeration)object);
				return ReflectionObjectModel.getInstance(object);
			}
		}
		else
			return Models.EMPTY_SCALAR;
	}

	/**
	 * Attempts to unwrap a model into underlying object. It can unwrap 
	 * {@link ReflectionObjectModelBase} and {@link ReflectionScalarModel} instances,
	 * as well a generic {@link TemplateScalarModel} into a String. All other
	 * objects are returned unchanged.
	 */
	public static final Object unwrap(Object model)
	throws
		TemplateModelException
	{
		if(model instanceof ReflectionObjectModelBase)
			return ((ReflectionObjectModelBase)model).getObject();
		if(model instanceof ReflectionScalarModel)
			return ((ReflectionScalarModel)model).getAsObject();
		if(model instanceof BooleanScalar)
			return BooleanScalar.getBoolean(model) ? Boolean.TRUE : Boolean.FALSE;
		if(model instanceof TemplateScalarModel)
			return ((TemplateScalarModel)model).getAsString();
		else return model;
	}

	/**
	 * For a given method, retrieves its publicly accessible counterpart. If a
	 * public method is declared by a non-public class, calling it throws an
	 * IllegalAccessException. This method will look for a method with same name
	 * and signature declared in a public superclass or implemented interface of this 
	 * method's declaring class. This counterpart method is publicly callable.
	 * @param method a method whose publicly callable counterpart is requested.
	 * @return the publicly callable counterpart method. Note that if the parameter
	 * method is itself declared by a public class, this method is an identity
	 * function.
	 * @throws NoSuchMethodException if an appropriate method could not be located.
	 */
	static final Method getPublicMethod(Method method)
	throws
		NoSuchMethodException
	{
		Class clazz = method.getDeclaringClass();
		// Short circuit for (hopefully the majority of) cases where the declaring
		// class is public.
		if((clazz.getModifiers() & Modifier.PUBLIC) != 0)
			return method;
		
		synchronized(publicMethodCache)
		{
			// Look first in the cache
			Method pubmethod = (Method)publicMethodCache.get(method);
			// If not cached ...
			if(pubmethod == null)
			{
				// ...lookup, and ...
				pubmethod = getPublicMethod(clazz, method.getName(), method.getParameterTypes());
				// ...cache.
				publicMethodCache.put(method, pubmethod);
			}
			return pubmethod;
		}
	}
	
	static final Method getPublicMethodOrNull(Method method)
	{
		if(method == null)
			return null;
		try
		{
			return getPublicMethod(method);
		}
		catch(NoSuchMethodException e)
		{
			return null;
		}
	}
	
	/**
	 * Looks up the method with specified name and signature in the first public
	 * superclass or implemented interface of the class. This method is safe to call
	 * from reflection, and will not throw IllegalAccessException.
	 * @param class the class whose method is sought
	 * @param name the name of the method
	 * @param paramTypes the classes of method parameters
	 */
	private final static Method getPublicMethod(Class clazz, String name, Class[] paramTypes)
	throws
		NoSuchMethodException
	{
		if((clazz.getModifiers() & Modifier.PUBLIC) != 0)
			return clazz.getMethod(name, paramTypes);
		
		Class superclazz = clazz.getSuperclass();
		if(superclazz != null)
		{
			try
			{
				return getPublicMethod(superclazz, name, paramTypes);
			}
			catch(NoSuchMethodException e)
			{
				// Intentionally ignored
			}
		}
		
		Class[] interfaces = clazz.getInterfaces();
		for(int i = 0; i < interfaces.length; ++i)
		{
			try
			{
				return getPublicMethod(interfaces[i], name, paramTypes);
			}
			catch(NoSuchMethodException e)
			{
				// Intentionally ignored
			}
		}
		
		throw new NoSuchMethodException();
	}
}
