/*
 * ====================================================================
 *
 * Copyright (c) 2000 Attila Szegedi.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes the Expose library
 *        (http://www.szegedi.org/expose)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Expose" and "Attila Szegedi" must not be used to endorse 
 *    or promote products derived from this software without prior written
 *    permission. For written permission, please contact expose@szegedi.org.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR ITS CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 */

package org.szegedi.expose.model;

import freemarker.template.*;

import java.lang.reflect.*;
import java.io.*;

/**
 * A class that will wrap a reflected parameterless method call into a
 * {@link TemplateScalarModel} interface. It can be used directly, and is
 * heavily used by {@link ReflectionObjectModel} to wrap properties of primitive
 * types.
 * @author Attila Szegedi, attila@szegedi.org
 * @version 1.0
 */

public final class ReflectionScalarModel
implements
	TemplateScalarModel
{
	private Object wrap;
	private Method getter;

	/**
	 * Creates a new model.
	 * @param object the object on which the method will be called.
	 * Can be <tt>null</tt> for static methods.
	 * @param getter the method that will be called to obtain
	 * the value of the model.
	 */
	public ReflectionScalarModel(Object object, Method getter)
	{
		this.wrap = object;
		this.getter = getter;
	}

	/**
	 * Calls the wrapped method, and calls toString() on its return value.
	 */
	public String getAsString()
	throws
		TemplateModelException
	{
		Object obj = getAsObject();
		return obj == null ? "null" : obj.toString();
	}

	/**
	 * Invokes the wrapped method, returns its return value.
	 */
	public Object getAsObject()
	throws
		TemplateModelException
	{
		try
		{
			return getter.invoke(wrap, null);
		}
		catch(Exception e)
		{
			StringWriter sw = new StringWriter();
			PrintWriter pw = new PrintWriter(sw);
			e.printStackTrace(pw);
			throw new TemplateModelException(sw.toString());
		}
	}

  /**
   * Returns true only if the property value is a null, or a Boolean with false value.
   * This allows boolean values to be used in <tt>if</tt> statement without testing for
   * <tt>obj == "true"</tt in the template>.
   */
	public boolean isEmpty()
  throws
    TemplateModelException
	{
	  Object obj = getAsObject();
    return obj == null || (obj instanceof Boolean && !((Boolean)obj).booleanValue());
	}
}
