/*
 * ====================================================================
 *
 * Copyright (c) 2000 Attila Szegedi.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes the Expose library
 *        (http://www.szegedi.org/expose)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Expose" and "Attila Szegedi" must not be used to endorse 
 *    or promote products derived from this software without prior written
 *    permission. For written permission, please contact expose@szegedi.org.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR ITS CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 */

package org.szegedi.expose.model;

import org.szegedi.collection.IdentityHashMap;

import freemarker.template.*;
import java.lang.ref.*;
import java.util.*;

/**
 * Base class for all classes that wrap objects into template models.
 * Instances of this class and descendant classes can be created directly, 
 * or utilizing a model cache through <tt>getInstance()</tt> calls, or polimorphically
 * through the {@link ReflectionUtilities#wrap(Object)} method.
 * @author Attila Szegedi, attila@szegedi.org
 * @version 1.0
 */
public class ReflectionObjectModelBase
implements
	TemplateScalarModel
{
	/** Returned from {@link #getType()} if the object is a ReflectionObjectModelBase */
	public static final int TYPE_OBJECT_BASE = -1;
	/** Returned from {@link #getType()} if the object is a ReflectionObjectModel */
	public static final int TYPE_OBJECT = 0;
	/** Returned from {@link #getType()} if the object is a ReflectionArrayModel */
	public static final int TYPE_ARRAY = 1;
	/** Returned from {@link #getType()} if the object is a ReflectionCollectionModel */
	public static final int TYPE_COLLECTION = 2;
	/** Returned from {@link #getType()} if the object is a ReflectionMapModel */
	public static final int TYPE_MAP = 3;
	/** Returned from {@link #getType()} if the object is a ReflectionIteratorModel */
	public static final int TYPE_ITERATOR = 4;
	/** Returned from {@link #getType()} if the object is a ReflectionEnumerationModel */
	public static final int TYPE_ENUMERATION = 5;
	/** Returned from {@link #getType()} if the object is a ResourceBundleHashModel */
	public static final int TYPE_RESOURCE_BUNDLE = 6;

	static final boolean noCache = Boolean.getBoolean("expose.reflection.nocache");
	private static final Map modelCache = noCache ? null : new IdentityHashMap();
	private static final ReferenceQueue refQueue = noCache ? null : new ReferenceQueue();

	Object object;

	ReflectionObjectModelBase(Object object)
	{
		this.object = object;
	}

	/**
	 * Creates a new model that wraps the specified String object.
	 * Note that since the only Freemarker-specific method in this class is
	 * <tt>getAsString</tt> there is no sense in creating an object of this
	 * class directly with anything except a String. If you need to wrap an object
	 * of another class, use one of specialized subclasses, or delegate the choice
	 * to {@link ReflectionUtilities#wrap(Object)}.
	 * @param object the String to wrap into a model.
	 */
	ReflectionObjectModelBase(String object)
	{
		this.object = object;
	}

	/**
	 * Returns a model wrapping the specified String object. If there is already
	 * a cached model instance for this String, returns the cached model instance.
	 * Models are cached using {@link WeakReference} objects. The caching can be turned
	 * off by setting the <tt>expose.reflection.nocache</tt> system property to
	 * true. In this case calling this method is equivalent to constructing a new model.
	 * Note that since the only Freemarker-specific method in this class is
	 * <tt>getAsString</tt> there is no sense in creating an object of this
	 * class directly with anything except a String. If you need to wrap an object
	 * of another class, use one of specialized subclasses, or delegate the choice
	 * to {@link ReflectionUtilities#wrap(Object)}.
	 * @param object the String to wrap into a model.
	 * @return the model for the String
	 */
	public static final ReflectionObjectModelBase getInstance(String object)
	{
		if(noCache)
			return new ReflectionObjectModel(object);
		
		ReflectionObjectModelBase model = lookup(object);
		if(model == null)
		{
			model = new ReflectionObjectModelBase(object);
			register(model, object);
		}
		return model;
	}

	/**
	 * Returns the wrapped object.
	 */
	public Object getObject()
	{
		return object;
	}

	final static ReflectionObjectModelBase lookup(Object object)
	{
		ReflectionObjectModelBase model = null;
		ModelReference ref = null;
		// NOTE: we're doing minimal synchronizations -- which can lead to
		// duplicate wrapper creation. However, this has no harmful side-effects and
		// is a lesser performance hit.
		synchronized(modelCache)
		{
			ref = (ModelReference)modelCache.get(object);
		}

		if(ref != null)
			return ref.getModel();
		
		return null;
	}
	
	final static void register(ReflectionObjectModelBase model, Object object)
	{
		synchronized(modelCache)
		{
			// Remove cleared references
			for(;;)
			{
				ModelReference queuedRef = (ModelReference)refQueue.poll();
				if(queuedRef == null)
					break;
				modelCache.remove(queuedRef.object);
			}
			// Register new reference
			modelCache.put(object, new ModelReference(model));
		}
	}

	/**
	 * Returns the type of this object (which is TYPE_OBJECT_BASE)
	 */
	public int getType()
	{
		return TYPE_OBJECT_BASE;
	}

	/**
	 * Returns either the toString() of the wrapped object or the string "null",
	 * if the wrapped object is null.
	 */
	public String getAsString()
	{
		return object == null ? "null" : object.toString();
	}

	/**
	 * Tells whether the model is empty. It is empty if either the wrapped object is
	 * null, or it is a Boolean with false value.
	 */
	public boolean isEmpty()
	{
		return object == null || (object instanceof Boolean && !((Boolean)object).booleanValue());
	}

	/**
	 * A special weak reference that is registered in the modelCache.
	 * When it gets cleared (that is, the model became unreachable)
	 * it will remove itself from the model cache.
	 */
	private static final class ModelReference extends WeakReference
	{
		Object object;

		ModelReference(ReflectionObjectModelBase ref)
		{
			super(ref, refQueue);
			object = ref.object;
		}

		ReflectionObjectModelBase getModel()
		{
			ReflectionObjectModelBase model = (ReflectionObjectModelBase)this.get();
			return model;
		}
	}

	private static String print(Object object)
	{
		if(object == null)
			return null;
		return object.getClass().getName() + "@" + Integer.toHexString(System.identityHashCode(object));
	}
}
